#set math.equation(numbering: "1.") // allow math equation referencing
#set heading(numbering: "1.") // heading numbering
#set text(font: "New Computer Modern") // nicer font
#set page(paper: "a4", numbering: "1") // page numbering
#let erfc = math.op("erfc") // erfc as math op

#import "@preview/cetz:0.3.4" // for diagrams
#import "@preview/dashy-todo:0.0.2": todo // for todo: TODO: remove when done

#let author = "Dimas Leenman"
#let title = "A Practical and Fast Atmospheric Scattering Model"
#let supervisor_fst = "Peter Vangorp"
#let supervisor_snd = "Alex Telea"

// FLIP with the reverse F
#show "nvidia-flip": "\u{A7FB}LIP"

// recall something
#let recall(reference) = {
  context {
    // don't take the body, so we can copy the entire figure
    query(reference).at(0)
  }
}

#let abstract = [Previous models of atmospheric scattering have chosen between visual accuracy,
  runtime performance, and implementation complexity,
  sometimes with constraints to only view from or low to the surface.
  Some of these models require precomputing some amount of data to a texture,
  trading implementation complexity and memory use for runtime performance.

  This work presents a new fitted model, intended for real-time rendering,
  that does not require precomputation. Unlike previous fitted models, the new
  model supports both ground and space views, and does not need to be refitted
  when the parameters of the atmosphere are changed.

  The new model is compared to existing models, as well as a path traced ground truth,
  and its output is measured to have a low visual error, using various distance metrics]

#set document(author: author, title: title, description: abstract)

#align(center, text(24pt)[*#title*])
#grid(
  columns: (4fr, 4fr),
  [_Author_ \ #author],
  [_First Supervisor_ \ #supervisor_fst \ \ _Second Supervisor_ \ #supervisor_snd],
)

#pagebreak()
= Abstract
#todo("Make neater")
#abstract

#outline()
#pagebreak()

= Introduction
As games become increasingly more realistic, more accurate visual
effects are needed to render them. In games such as Microsoft Flight
Simulator @fs2024, DCS @DCS, and the Scatterer mod for Kerbal Space
Program @ksp-scatterer the atmosphere is an important element to get
right. Other games, such as Gran Turismo 7 @gt7 make use of a complex
sky simulation to ensure both the sky itself and resulting lighting
looks correct. This work aims to create a new method of rendering the
atmosphere, that does not rely on a large amount of lookup tables found
in previous implementations, while not sacrificing accuracy or
performance.

= Atmospheric scattering
<chap:atmospheric-scattering>
All particles in the atmosphere of the earth affect how light is
scattered inside it. When a photon hits a particle, it can either be
scattered into a new direction, or absorbed. Which of these events
happen, depends on the wavelength of the photon, as well as the type and
size of the particle. For every photon, these events can happen 0 times,
1 time, referred to as single scattering, or more than once referred to
as multiple scattering.

Before going over previous work, it may prove useful to provide a standard
set of symbols, coordinates, as well as a short
introduction on the physics behind atmospheric scattering.

== Coordinate system
The figure below shows the coordinates used for a viewer $v$ at height $h$
in an atmosphere, with light coming from light source $l$. The coordinate
system for this is described in @tab:coord-sys.

#figure(caption: [Viewer $v$ in an atmosphere.], cetz.canvas({
  import cetz.draw: *
  // planet
  arc-through((-5, -3), (0, 0), (5, -3))

  // planet radius
  line((0, 0), (0, -1.5))
  stroke((dash: "dashed"))
  line((0, -1.5), (0, -3))
  arc-through((-8.5, -2), (0, 3), (8.5, -2)) // atmosphere
  line((0, 1), (0, 3)) // surface normal
  stroke((dash: none))
  content((1em, -0.75), [$r$])

  // viewer
  line((0, 0), (0, 1))
  content((1em, 0.5), [$h$])

  // view ray
  line((0, 1), (4, 3))

  // view angle
  arc((0, 2), start: 90deg, delta: -63.43deg, radius: 1)
  content((1em, 1.5), [$theta_v$])

  // sun ray
  stroke((dash: "dashed"))
  line((0, 1), (-4, 3))
  stroke((dash: none))

  // sun angle
  arc((0, 2), start: 90deg, delta: 63.43deg, radius: 1)
  content((-1em, 1.5), [$theta_l$])

  // sun-view angle
  content((1em, 2.3), [$gamma$])

  // sun
  stroke((dash: "dashed"))
  circle((-4, 3), radius: 0.2)
  content((-4.4, 2.6), [$l$])

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((0, 1), radius: 0.1)
  content((-1em, 0.9), [$v$])

  // ray labels
  content((-2.5, 2), [$p_l$])
  content((2.5, 2), [$p_v$])
}))
<fig:angles>

#figure(
  caption: "Coordinates",
  table(
    columns: 2,
    align: (col, row) => (center, center).at(col),
    inset: 6pt,
    [Radius $r$], [The radius of the planet],
    [Height $h$], [The height viewer $v$ is from the surface],
    [View zenith $theta_v$], [The angle between the surface normal, and view direction],
    [Sun zenith $theta_l$], [The angle between the surface normal, and the light direction],
    [Angle $gamma$], [The angle between the view direction, and light direction],
    stroke: 0.5pt
  ),
)
<tab:coord-sys>

== Physics of scattering
This section provides a basic overview of the physics behind a photon
scattering in an atmosphere. A possible path of a photon scattering from
light source $l$ towards viewer $v$ can be seen below.

#figure(caption: [Viewer $v$ in an atmosphere.], cetz.canvas({
  import cetz.draw: *
  // planet
  arc-through((-5, -1), (0, 0), (5, -1))

  // planet radius
  stroke((dash: "dashed"))
  arc-through((-7.5, 1), (0, 3), (7.5, 1)) // atmosphere
  stroke((dash: none))

  // view to scatter
  line((-3, 1), (1, 1))

  // scatter to sun
  line((1, 1), (5, 4))

  // scatter angle
  arc((0.5, 1), start: 180deg, delta: -145deg, radius: 0.5)
  content((0.75, 1.8), [$theta_s$])

  // scatter to sun, multiple
  stroke((dash: "dashed"))
  line((1, 1), (2, 1))
  line((2, 1), (3, 0))
  line((3, 0), (5, 4))

  // sun
  circle((5, 4), radius: 0.2)
  content((5.4, 3.6), [$l$])

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((-3, 1), radius: 0.1)

  // viewer
  content((-3.3, 0.7), [$v$])

  // path a
  content((-0.75, 0.8), [$a$])

  // path b
  content((3.9, 1), [$b$])
}))
<fig:angles>

Here, path $a$ represents a path with only one scatter event, where the photon
scatters at an angle $theta_s$. Path $b$ represents a path with multiple
scattering events.

When light travels through the atmosphere, it is attenuated based on the
density of the atmosphere it travels through. Given a density function
$rho(h)$ that depends on height $h$ from the surface, the _optical depth $tau$_
from viewer $v$, with direction $theta_v$ can then be calculated with the
following integral:

$ tau = integral_a^b rho(sqrt(t^2 + h^2 + 2 h t cos theta_v)) d t $
<eq:optical-depth>

Where $a$ and $b$ are the start and end points of the ray respectively.
For $rho(h)$, most previous work discussed in @sec:previous-work[Chapter] uses
the exponential function for the density of the medium:

$ rho(h) = exp(-h / h_0) $

Here, $h_0$ is the _scale height_, which determines how fast the density
falls off based on height.

For ozone, @bruneton-new and @seb-skyatmo use the following density:

$ rho(h) = max(0, 1 - (|h - 25|) / 15) $

Where $h$ is in kilometers.

The _transmittance $T$_, which indicates how much light is attenuated, can then
be calculated as follows:

$ T = exp(-beta_a tau) $

Where $beta_a$ is the _absorption coefficient_, which determines how much
of the light is absorbed.

Transmittance $T$ represents how much light is attenuated for a single path
in the atmosphere, but does not take scattering into account. When a photon
scatters, it changes direction. The new direction is not uniform, and depends
on a phase function $F(theta_s)$. The phase function used depends on the
scattering medium. For Mie scattering, which is used to model aerosols,
a common phase function to use is the Cornette-Shanks phase function described
by @cornette-shanks. This phase function is used by @bruneton-atmo,
@production-volume-rendering, @nishita93 and others. It is shown below.

$
  F(theta_s) = 3 / (8pi) ((1-g^2)(1+cos^2 theta_s)) / ((2+g^2)(1+g^2 - 2g cos theta_s)^(3 / 2))
$
<eq:phase-cornette-shanks>

An alternative phase function used by @PBRT3e is the Henyey-Greenstein phase
function, described by @henyey-greenstein can be seen below:

$ F(theta_s) = 1 / (4pi) (1-g^2) / ((1 + g^2 + 2g cos theta_s)^(3 / 2)) $
<eq:phase-henyey-greenstein>

Here, $g$ determines how directional the phase function is, with values near 1
being more directional, and thus photons less likely to significantly change
direction upon scattering. For Rayleigh scattering, $g = 0$ is typically used.
@seb-skyatmo introduces an isotropic phase function, shown below:

$ F(theta_s) = 1 / (4pi) $

Using the phase function, it is now possible to calculate the transmittance
for the entire path that light can take through the scattering medium.
For a path with a single scattering event, the total luminance $L_"scatter"$
reaching the viewer can be calculated as follows:

$ L_"scatter" = E_"light" F(theta_s) beta_s rho(h_s) T_s T_l $
<eq:scatter-luminance>

Where $beta_s$ is the scattering density coefficient, $h_s$ is the height
at which the scatter event takes place. $T_s$ is the transmittance from the
viewer to the scattering point, $T_l$ is the transmittance from the scattering
point to the light source, and $E_"light"$ is the incident luminance coming
from the light source. Note that when the path between the scattering point
and the light source is obstructed, $E_"light" = 0$.

In order to take multiple media types into account, their transmittance must
be combined. This is done by multiplying all transmittance from all media types
together. $L_"scatter"$ calculates the resulting luminance from scattering for
only one medium type, and thus needs to be calculated for all media types as
well. An example of the total luminance $L_"total"$ due to scattering from
two scattering media $m$ and $r$, with respective phase functions,
transmittance, scattering densities and density functions can be seen below:

$
  L_"total" = E_"light" F^m (theta_s) beta^m_s rho^m (h_s) T^m_s T^r_s T^m_l T^r_l
  + E_"light" F^r (theta_s) beta^r_s rho^r (h_s) T^m_s T^r_s T^m_l T^r_l
$
<eq:scatter-luminance-total>

When calculating the luminance of a path with multiple scattering events,
@eq:scatter-luminance can be applied recursively, where $E_"light"$ is
replaced by another invocation of $L_"scatter"$.

= Path tracing
Path tracing is a brute force method that attempts to calculate all possible
paths a photon can take when scattering in an atmosphere. For this reason,
it is often used as a reference when implementing other models of atmospheric
scattering. @seb-skyatmo, @sky-comparison and @fitted-radiance use a path
tracer as a reference. @gt7, @hosek-wilkie and @preetham instead use it to fit
a model to the result of a path tracer.

The path tracer described here is the implementation provided by
@seb-skyatmo, and was tested against @PBRT3e for correctness.
The implementation is based on @production-volume-rendering.

As it is infeasible to compute all possible photon paths, the final image is
created using Monte-Carlo integration. This is done by averaging many
different, random photon paths into a final color value for the pixel.

== Finding the scattering position
In order to find a point at which a scattering or absorption event occurs,
delta tracking is used @delta-tracking. Delta tracking is an extension of
closed-form tracking @closed-form-tracking, which is used for homogeneous
volumes. Tracking allows importance sampling any homogeneous volume with
exponential attenuation according to Beer's law

$ T = exp(-beta rho(h) t) $

where $t$ is the length of the path the photon takes through the volume,
and $h$ is the height from the planet surface.


Tracking can then be used to determine the distance $t'$ that the photon
travels before being scattered, using @eq:tracking-dist, where $xi$ is a
uniform random number where $0 <= xi <= 1$.

$ t' = -ln(1 - xi) / beta $
<eq:tracking-dist>

As the atmosphere is modeled as an exponentially decaying density based on
height from the surface, the tracking method cannot be used there.
Instead, it is modified to take this density into account, which results in
delta tracking. When a photon scatters at distance $t'$ along its path, but
the density $beta rho$ at this point is $0$, no scattering can happen.
This is instead treated as a scattering event where the photon continues
to travel in the same direction. If the density $beta rho > 0$, delta tracking
only considers a scatter event to happen if @eq:delta-smaller holds, where
$beta_("max")$ is the highest density that can be found in the volume.

$ xi < (beta rho) / (beta rho_("max")) $
<eq:delta-smaller>

If this does not hold, delta tracking is performed again, using the predicted
scattering position derived from $t'$ as starting point. In case of absorption,
the photon path does not scatter any light, and thus no light is added, and no
further scattering events are considered. A minimal example of delta tracking
can be seen below:

#figure(
  caption: "Delta tracking",
  ```glsl
  vec3 delta_tracking(vec3 start, vec3 dir) {
      while (true) {
          float xi = random();
          float zeta = random();
          float t = -ln(1.0 - xi) / beta_max;
          if (zeta < beta * volume_density(start + dir * t) / beta_max) {
              // scatter event, return where
              return start + dir * t;
          } else if (length(start + dir * t) > max_distance) {
              // outside of volume, stop
              return start - dir;
          } else {
              start = start + dir * t;
          }
      }
  }
  ```,
)
<code:delta-tracking>

At the found scattering position, it is assumed that both a photon from the
light source scatter here, as well as a photon from another scattering event
in the atmosphere.

In order to combine multiple multiple types of media with different scattering
or absorption properties, $beta_"max"$ becomes the sum of all the $beta$ of
the different types of volumes. Then, when testing for a scattering event,
each of the $beta$ is assigned an interval. If $xi$ is within that interval,
there is a scattering event or absorption event for that medium.

== Light contribution
The contribution of light to this scattering event can be computed in several
ways. It is possible to perform delta tracking towards the light source, and
if no scattering event happens, the light contribution is equal to
$F(theta_s)$. It is also possible to compute transmittance $T$ instead,
calculating the optical depth using numerical integration directly,
using a lookup table, or using ratio tracking, as described by
@residual-ratio-tracking.

Ratio tracking is a modification of delta tracking, where instead of testing
whether there is a scattering event at each predicted scattering event at
distance $t`$, the attenuation $T$, initialized with $1$, is multiplied by
@eq:ratio-tracking-mul

$ 1 - (beta rho(h)) / (beta_("max")) $
<eq:ratio-tracking-mul>

The resulting transmittance is then multiplied by $F(theta_s)$, and added to the
final color of this path. A minimal example of ratio tracking is as follows:

#figure(
  caption: "Ratio tracking",
  ```glsl
  float ratio_tracking(vec3 start, vec3 dir) {
      float T = 1.0;
      while (true) {
          float xi = random();
          float t = -ln(1.0 - xi) / beta_max;
          start = start + dir * t;
          if (length(start) > max_distance) {
              // outside of volume, stop
              break;
          }
          T *= 1.0 - beta * volume_density(start) / beta_max;
      }

      return T;
  }
  ```,
)
<code:ratio-tracking>

Performing these steps once results in single scattering.

== Phase function
In order to perform multiple scattering, these steps are repeated with a new
direction for the photon path, up to $N$ times, where $N$ is the scattering
order. This new direction is chosen by importance sampling the phase function.
For Rayleigh scattering, no importance sampling is performed, as it is assumed
that the phase function is isotropic.
For Mie scattering, the Henyey-Greenstein phase function is used instead.
@PBRT3e explains how importance sampling is performed for this phase function
in _chapter 15.2.3: Sampling phase Functions_.

== Final color
As each path is calculated for a single wavelength of light, it needs to be
repeated for each wavelength. To then eliminate noise in the image, this needs
to be repeated several times as well.
The implementation provided by @seb-skyatmo simply selects the red,
green, or blue color channel at random, by drawing from a uniform random
distribution. Based on the color channel that was selected, different
values for the scattering density $beta$ are selected. The other two color
channels are then set to 0 for this contribution. This is then repeated
many times, and the average of this process is used as final color for the
specified pixel.

= Previous work <sec:previous-work>
A number of methods have been developed for rendering atmospheric scattering
in real time. This section provides an overview of various methods that have
been developed for this purpose, as well as how these methods relate to each
other.

== Single scattering
The work discussed in this section covers implementations of the physics
discussed in @chap:atmospheric-scattering[Chapter], assuming only single
scattering occurs. Most of the focus is put on efficiently evaluating the
optical depth $tau$, from @eq:optical-depth, as using numerical integration
of it results in a large amount of evaluations of $rho(h)$. This can quickly
become prohibitively expensive.

One of the earlier models for atmospheric scattering was developed in
@nishita93, in order to view an atmosphere from space. @gpu-gems-atmo later
shows that this technique can be adapted to work from a ground view as well.

In order to keep the computational cost down, @nishita93 assumes that multiple
scattering can be ignored as it has a negligible effect. They also assume
absorption due to ozone is negligible, and that the density of the air falls
off exponentially, as follows:

$ rho(h) = exp(-h / h_0) $

@nishita93 assumes only two types of scattering can happen. Rayleigh
scattering, for smaller particles, and Mie scattering, for larger particles.
For both types of scattering, they opt to use the Cornette-Shanks phase
function as described in @cornette-shanks:

#recall(<eq:phase-cornette-shanks>)

For Rayleigh scattering, they assume $g = 0$. For Mie scattering, it is
assumed that$0.7 <= g <= 0.85$. The scale heights $h_0$ are set to
$h^r_0 = 7994m$ and $h^m_0 = 1200m$ for Rayleigh and Mie respectively.

In order to reduce computational complexity, @nishita93 precomputes the
optical depth, and stores it in a 2D lookup table. Recall the equation for
the optical depth:

#recall(<eq:optical-depth>)

When assuming that the integral starts at the viewer's position on the view
ray, and ends at infinity, and the air density falls of exponentially, it can
be rewritten as follows:

$ tau = integral_0^infinity exp(sqrt(t^2 + h^2 + 2 h t cos theta_v) / h_0) d t $

As the scale height does not change often, it can be assumed that the integral
now only depends on the height of the viewer $h$, and the view direction
$theta_v$. This allows these parameters to be used as axes in the 2D lookup
table.

For the height, $N$ values of $h_i$ are chosen, with a spacing of

$ h_i = h_0 log((1 - i) / N) + h $

Where $h_i$ is the height used to compute the $i$th row of the lookup table.
Then, the columns of the lookup table, which depend on the viewing angle,
also have to be computed. This is done by taking the heights $h_i$, and
imagining them as spheres around the center of the planet. Then, a number of
cylinders, aligned to the light direction is swept through the spheres.
The index of the sphere, and the index of the cylinder are then used to look
up the row and column of the lookup table.

To calculate the optical depth, trapezoidal integration is used.

In order to still be compute the optical depth for a segment of the view path
instead of from the viewer to infinity, the optical depth for the end of the
path to infinity can be subtracted from the optical depth from the viewer to
infinity.

To arrive at a final color for the red, green, and blue color channels,
the equation shown in @eq:scatter-luminance-total is evaluated for a number
of points along the view path, and added together.

#recall(<eq:scatter-luminance-total>)

This is then done for every pixel in the output image.

The transmittance for the light direction is calculated using the optical
depth stored in the lookup table. The transmittance for the view direction
is calculated using trapezoidal integration, as this can be performed at the
same time the luminance is accumulated.

From here on, the work of @nishita93 remains used as a base for single
scattering. Most of the variations presented next vary from their
implementation in the manner of how they implement the numerical integration
of the optical depth integral.

@oneil-previous presents a different formulation of the lookup table that
@nishita93 uses. Instead of using the spheres intersecting the cylinders,
it is possible to simply divide the height between $h = 0$ and $h = r_a$, where
$r_a$ is a defined radius of the atmosphere. Then when looking up the value,
@oneil-previous linearly interpolates between the entries in the lookup table
for the specific height. For the view angle, they use the result of
$cos theta_v$ to map to a column in the lookup table.

Besides storing the optical depth for Rayleigh and Mie, @oneil-previous also
stores the density $rho(h)$ for Rayleigh and Mie in the lookup table. If the
view path for the optical depth intersects the planet, this is set to $0$,
which emulates the effect of a planet shadow.

Then, in order to compute the transmittance of the view path segment, the
lookup table is used again, and the segment optical depth is calculated the
same way as @nishita93 does it. However, if the view path intersects the
planet, the view direction the lookup table is sampled with is rotated by
$180degree$, and the order in which they are subtracted is swapped. This
avoids floating point precision errors due to a very large optical depth.

Later, @gpu-gems-atmo modifies their previous work in @oneil-previous to run
on the GPU. Instead of performing the full calculation for each pixel, they
move it to the vertex shader, and let the GPU interpolate between vertices to
get the final per pixel result. The phase functions $F(theta_s)$ are however
calculated per pixel, as this resolves some artifacts.

As GPUs at the time did not have the ability to read from a texture from the
vertex shader, it is not possible to use a lookup table. Instead,
@gpu-gems-atmo finds that it is possible to approximate the optical depth
using a different formula. For the height, they find that $exp(-4h)$ is a
common factor for all view angles.
For the view angles, they find that taking the logarithm of the view angle
results in an unknown curve, and fit a polynomial to this curve.
This results in the following equation:

$ tau = exp (- s_1 h) s_2 exp (5.25x^4 - 6.80x^3 + 3.83x^2 + 0.459x - 0.00287) $

Where $s_1$ and $s_2$ are constants that depend on the atmosphere parameters.
Of note is that this equation for optical depth only works for one combination
of planet radius $r$ and scale height $h_0$. If these are changed, the
polynomial and $s_1$ and $s_2$ have to be fit again. Using this, @gpu-gems-atmo
implements their single scattering the same way as described in
@oneil-previous.

Besides the polynomial approximation @gpu-gems-atmo finds for the optical
depth, there has also been work in physics literature to solve a similar
problem. The result is the Chapman function $"Ch"(x, chi)$, first introduced
in @chapman. As this function cannot be computed exactly, other approximations
have been formulated, such as by @kocifaj, @chapman-approx-big. A more
general overview can be found in @vasylyev-chapman.

@gpu-pro-3 derives a new approximation of the Chapman function, that is
simpler and more numerically stable when implemented using 32 bit floating
point. @gpu-pro-3 first starts with the Chapman function presented in @kocifaj:

$
  "Ch"(x , chi) = 1 / 2 [cos chi + exp (frac(x cos^2 chi, 2)) erfc (sqrt(frac(x cos^2 chi, 2))) (1 / x + 2 - cos^2 chi) sqrt(frac(pi x, 2))]
$
<eq:chap-kocifaj>

Where

$ x = (r + h) / h_0, quad "and" quad chi = theta_v $

The optical depth from viewer $v$ can then be calculated as follows:

$ tau = "Ch"((r + h) / h_0, theta_v)exp(-h / h_0) $
<eq:optical-depth-chapman>

In order to simplify @eq:chap-kocifaj, @gpu-pro-3 makes the following
assumptions:

$
  "Ch"(x, chi) = "Ch"(x, -chi), quad "Ch"(x, 0) = 1, quad "and" quad lim_(x arrow.r oo) "Ch"(x, chi) = 1 / (cos chi)
$

Using this, they find an approximation that works well when $chi < 90 degree$:

$ "Ch"'(c, chi) approx c / (1 + (c - 1) cos chi) quad "if" chi < 90 degree $
<eq:chapman-lt-90>

where $c = "Ch"(x, 90 degree)$. In order to then derive the Chapman function
for $chi > 90 degree$, it is possible to restructure the problem. The optical
depth from the viewer to infinity is the same as the optical depth from
infinity behind the viewer to infinity in front of the viewer. Then, by
subtracting the optical depth from the viewer to infinity behind the viewer,
the optical depth from the viewer to infinity can be calculated. This is
expressed as follows:

$
  "Ch"(x, chi) = 2 exp(x - x sin chi) "Ch"(x, 90degree) - "Ch"(x, 180degree - chi)
$
<eq:chapman-gt-90>

This means, if $"Ch"(x, chi)$ is known for $chi < 90degree$, it is also known
for $chi >= 90degree$. Rewriting @eq:optical-depth-chapman, @eq:chapman-lt-90,
and @eq:chapman-gt-90, as it is desired to calculate the optical depth
without precision errors, results in the following equation for optical depth:

$
  tau_"Ch" = cases(
    display(& exp(-h / h_0) c / (1 + c cos theta_v) & "if" theta_v < 90 degree),
    display(- & exp(-h / h_0) c / (1 - c cos theta_v) + 2 sqrt(x_0) exp(r / h_0 - x_0) quad & "otherwise")
  )
$

Where

$
  c = sqrt((r + h) / h_0), quad "and" quad x_0 = sqrt(1 - cos^2 theta_v)((r + h) / h_0)
$

@gpu-pro-3 then uses $T_"Ch"$ for the optical depth to implement a single
scattering atmosphere, similar to how @nishita93, and @oneil-previous have done
it.

Of note is another approximation to optical depth by @bruneton-atmo. While not
explained is the paper, they implement an approximation of the optical depth
in the original implementation, used for the transmittance when calculating
the luminance of reflected light of the surface of the planet.
Recall the equation for optical depth:

#recall(<eq:optical-depth>)

Here, the distance from the surface at position $t$ is calculated using:

$ h_t = sqrt(t^2 + h^2 + 2h t cos theta_v) $

By assuming that

$ sqrt(1 + u) approx 1 + u / 2 $

This allows rewriting the optical depth integral, with
$rho(h) = exp(-h / h_0)$, and the path going from 0 to maximum distance $d$,
to the following:

$ tau approx integral_0^d exp(-(t^2 / (2 h) + t cos theta_v) / h_0) d t $

The primitive of this integral is as follows:

$
  Tau(t) approx exp((h cos^2 theta_v) / (2 h_0)) erfc((t + h cos theta_v) / sqrt(2 h dot h_0)) sqrt(pi / 2 dot h dot h_0)
$

$erfc(x)$ can be approximated with

$ erfc(x) approx 2 exp(-x^2) / (2.3192x + sqrt(1.52x^2 + 4)) quad "if" x >= 0 $

When $x < 0$, $erfc(-x) = 2 - erfc(x)$ can be used.
Combining this allows the optical depth to be evaluated from the viewer up to
distance $d$ with the following code:

#figure(
  ```glsl
  float optical_depth(float h_0, float h, float cos_theta_v, float d) {
      float a = sqrt((0.5 / h_0) * h);
      vec2 a01 = a * vec2(cos_theta_v, cos_theta_v + d / h);
      vec2 a01s = sign(a01);
      vec2 a01sq = a01 * a01;
      float x = a01s.y > a01s.x ? exp(a01sq.x) : 0.0;
      vec2 y = a01s / (2.3193 * abs(a01) + sqrt(1.52 * a01sq + 4.0))
          * vec2(1.0, exp(-d / h_0 * (d / (2.0 * r) + cos_theta_v)));
      return sqrt((6.2831 * h_0) * h) * exp((r - h) / h_0)
          * (x + dot(y, vec2(1.0, -1.0)));
  }
  ```,
  caption: [@bruneton-atmo's optical depth approximation],
)

/* original code:
#figure(
    ```glsl
    // optical depth for ray (r,mu) of length d, using analytic formula
    // (mu=cos(view zenith angle)), intersections with ground ignored
    // H=height scale of exponential density function
    float opticalDepth(float H, float r, float mu, float d) {
        float a = sqrt((0.5/H)*r);
        vec2 a01 = a*vec2(mu, mu + d / r);
        vec2 a01s = sign(a01);
        vec2 a01sq = a01*a01;
        float x = a01s.y > a01s.x ? exp(a01sq.x) : 0.0;
        vec2 y = a01s / (2.3193*abs(a01) + sqrt(1.52*a01sq + 4.0))
            * vec2(1.0, exp(-d/H*(d/(2.0*r)+mu)));
        return sqrt((6.2831*H)*r) * exp((Rg-r)/H)
            * (x + dot(y, vec2(1.0, -1.0)));
    }
    ```,
    caption: [Optical depth approximation]
)
<code:bruneton-optical-depth>

The email:
opticalDepth(float H, float r, float mu, float d) computes

D = integral from 0 to d of exp(-(r(t)-Rg)/H) dt

where r(t) = sqrt(r^2+2.t.r.mu+t^2), which, using sqrt(1+u) ~= 1+u/2 for
u small, can be approximated with r(t) ~= r + t.mu + t^2/(2.r).

Replacing this in D gives

D ~= exp((Rg-r)/H) * integral from 0 to d of exp(-(t.mu+t^2/(2r))/H) dt

A primitive of the integrand is (see Wolfram Alpha)

sqrt(pi.H.r/2).exp(mu^2.r/(2.H)).erfc((mu.r+t)/sqrt(2.H.r)) + cste

and

erfc(x) ~= 2.exp(-x^2)/(2.3192*x+sqrt(1.52*x^2+4)) for x >= 0

Using also erfc(-x)=2-erfc(x), this gives the formula used in this function.

*/

== Multiple scattering
Besides single scattering, multiple scattering adds a non-negligible
contribution to the total luminance of the scattered light. On earth, this is
best noticed during sunset and sunrise, as it adds an extra blue glow to the
atmosphere.

When assuming single scattering, the chosen scattering direction is trivial,
as this direction changes from the light directly towards the viewer when a
photon is scattered. When assuming two scattering events per photon path,
this direction is less trivial. The second scatter, where the photon scatters
towards the viewer, now has to integrate over all scattering directions to
account for all the incoming luminance due to scattering.

When using numerical integration, this results in having to evaluate many
single scattering paths for one multiple scattering event. This becomes
prohibitively expensive to compute. In order to still render an atmosphere in
real time, @bruneton-atmo proposes using several lookup tables.

In order to compute the transmittance, @bruneton-atmo uses a lookup table.
Instead of storing the optical depth, and using it to calculate the
transmittance, they store the transmittance directly in the table.
This allows storing the combined transmittance of all scattering media in a single
table, instead of one table for the optical depth of each media type.

Instead of evaluating the single scattering equation directly, @bruneton-atmo
recognizes it can be precomputed as well. Recall the single scattering
equation:

#recall(<eq:scatter-luminance>)

Here, $T_s$ depends on the viewer height $h$ and view direction $theta_v$.
$T_l$ depends on the light direction $theta_l$. The height of the scattering
event can be calculated from $h$ and $theta_v$, as many scattering events
happen along one view ray. The last dependency is the scattering angle $theta_s$.
This means single scattering only relies on 4 different variables, and can thus
be precomputed. The results are stored in a 4D lookup table.

As 4D textures are not available on the GPU, @bruneton-atmo instead chooses to
use a 3D texture, and for each slice of the texture, divide it into multiple
tiles to emulate a 4D texture.

Then, when $L_"scatter"$ is needed, it can simply be retrieved from the lookup
table. Rayleigh and Mie scattering are stored in separate tables.

The contribution of multiple scattering can be precomputed in a similar manner
to how it is done for single scattering. Instead of calculating the incoming
luminance from the sun due to transmittance, each sample point along the view
path integrates the result of single scattering over each direction. For more
than two scatter events, the previous result from the multiple scattering
table is read instead.

Integrating the scattering for all directions on all sample points along the
view path is expensive. To resolve this, @bruneton-atmo uses another lookup
table, that calculates the incoming luminance from multiple scattering for
every scattering direction $theta_s$, viewer height $h$ and light direction
$theta_l$. This lookup table can then be used to evaluate the incoming
luminance due to multiple scattering at each sample point.

This process is then repeated for several steps, in order to compute scattering
for more than 2 scatter events per path, referred to as _scattering orders_ by
@bruneton-atmo. This then results in a final lookup table that incorporates
both the single scattering and multiple scattering. This table is then read to
determine the luminance.

#todo[Replace luminance with radiance]

@elek presents a variation on the model presented by @bruneton-atmo. Instead of
using a table with 4 dimensions, @elek chooses to not incorporate the scattering
angle $theta_s$, and use a 3-dimensional table instead. This means that the
phase function has to be calculated, instead of reading it from the table when
computing luminance. It also means the planet shadow cannot be stored in the
table either. @elek argues this effect is negligible enough for this tradeoff
to work.

@seb-skyatmo proposes a different method of precomputing the
scattering. Instead of precomputing single scattering for all viewer positions,
they use a 2D lookup table, and recompute it when either light direction
$theta_l$ or viewer height $h$ changes.

For multiple scattering, @seb-skyatmo chooses to instead approximate
the result, instead of fully calculating each scattering order. To make the
approximation work, @seb-skyatmo assumes that after single scattering,
the phase function becomes isotropic. This eliminates the dependence on the
scattering direction $theta_s$ for multiple scattering.

Second order scattering can then be calculated by, for height $h$, and view
direction $theta_v$, calculate the incoming luminance from scattering for every
direction.

Scattering orders higher than 2 can be approximated by multiplying the second
order scattering by $F_"ms"$, which is a geometric series infinite sum:

$ F_"ms" = 1 + f_"ms" + f^2_"ms" + f^3_"ms" + ... = 1 / (1 - f_"ms") $

$f_"ms"$ represents the luminance that the sample point would receive, if the
atmosphere would emit light, with no other scattering contribution. $f_"ms"$
can thus be calculated by integrating the transmittance over all directions.

== Fitted models
The previously discussed methods have all tried computing the scattering
equations of @chap:atmospheric-scattering[Chapter] directly. This section
introduces a different method, that attempts to find an equation that can be
fitted to the output of a path tracer, in order to then render the atmosphere
without requiring the scattering equations, or precomputed lookup tables.
To keep complexity of the fitted model low, these do assume that the viewer is
either on the ground, or close to it, and thus do not support viewing from
space.

One of the earlier models to do this was presented by @preetham. Their model
is based on an earlier sky luminance model by @perez-formula:

$
  cal(F)(theta, gamma) = (1 + A exp(B / (cos theta)))(1 + C exp(D gamma) + E cos^2 gamma)
$

Here, $A$, $B$, $C$, $D$, $E$ are coefficients that are fitted to match a
reference. The presented model does not directly give luminance for a specific
wavelength, or red, green, and blue channels. Instead, the resulting color
space is _CIE $x y Y$_. This can then be converted to the luminance for a
specific wavelength as @preetham describes.

Luminance $Y$ is calculated as follows:

$ Y = Y_z (cal(F)(theta_v, gamma)) / (cal(F)(0, theta_l)) $

The chromaticity values $x$ and $y$ are calculated similarly:

$
  x = x_z (cal(F)(theta_v, gamma)) / (cal(F)(0, theta_l)) quad "and" quad
  y = y_z (cal(F)(theta_v, gamma)) / (cal(F)(0, theta_l))
$

Details for calculating $Y_z$, $x_z$ and $y_z$ are further described in
@preetham.

The coefficients of the model given in @preetham are calculated by fitting them
to the results of a path tracer. The result is a single formula that can model
the color of the atmosphere, given $h = 0$.

@hosek-wilkie improves upon the findings of @preetham, by taking the reflection
of light from the ground into account. They also improves the aureole,
which is a bright halo around the sun. This halo is also darkened when the sun
is near the horizon. This results in a new formula:

$
  & bb(F)(theta, gamma) =                                                      \
  & (1 + A exp(B / (0.01 + cos theta)))
    (C + D exp(gamma E) + F cos^2 gamma + G chi(H, gamma) + I sqrt(cos theta)) \
$

Where $A$, $B$, $C$, $D$, $E$, $F$, $G$, $H$, and $I$ are the coefficients to
fit. $chi$ is calculated as follows:

$ chi(g, alpha) = (1 + cos^2 alpha) / (1 + g^2 - 2g cos alpha) $

The final luminance for wavelength, or color channel $lambda$ is then
calculated with:

$ L_lambda = bb(F)(theta_v, gamma) L_(M lambda) $

All coefficients are then again fit on the result of a path tracer.
Unlike the work of @preetham, the coefficients of $bb(F)(theta, gamma)$ are
interpolated from a larger set of coefficients, based on turbidity, ground albedo,
and sun elevation $theta_s$.

While these models fit the general sky quite well, they are not capable of
representing more finer details. To address this, @fitted-radiance introduces
a new model with significantly more parameters. Besides outputting the direct
luminance, it also outputs the transmittance, and polarization of light.
Contrary to @preetham and @hosek-wilkie, this model is also able to represent
the sky when the sun is below the horizon, as well as $h > 0$, up to a maximum
height. The downside is that this requires significantly more coefficients,
with the largest version of the model being 2.2 GB in size.

As @bruneton-atmo and @seb-skyatmo have done with precomputing single and
multiple scattering, it is also possible to store the results of a path tracer
in a lookup table directly, instead of fitting a formula to the results.
This is the method @gt7 takes. For the lookup table, they use 64 2D lookup
tables, each table representing a single time of day, of angle $theta_l$,
separated in $theta_v$ and $gamma$. In order to fix artifacts near the horizon,
more tables are allocated when the sun is close to the horizon. Per table,
a special spherical mapping is used that increases the number of samples near
the sun.

= Comparing models
Before comparing atmospheres, it proves useful to decide on a method in order
to compare them. A basic method for doing this is using the Root-Mean Square
Error, or RMSE for short:

$ "RMSE" = sqrt(sum^N_(i = 0) (v_(i,m) - v_(i,t))^2 / N) $

Where $v_(i,m)$ is the $i$th sample of the model to test, and $v_(i,t)$ is the
$i$th sample of the reference model.

When comparing atmospheres, each of these samples may represent a different
combination of viewer height $h$, view direction $theta_v$, light direction
$theta_l$, and the difference between the light and view direction $gamma$.
Depending on How the model outputs color, the different color channels, or
different wavelengths of light are considered separate samples as well. No
post-processing, tonemapping, or other effects should be applied to the
luminance the models output.

As RMSE as described here operates on sets of samples, it is best suited for
comparing entire HDR images output from the atmosphere models. While it is
possible to use it to compare the per-pixel differences, better methods are
available.

nvidia-flip, introduced in @flip, aims at highlighting the differences between
images, by emulating how human perception notices differences when quickly
flipping between two images. It shows where the per-pixel differences are in an
error map, output after comparison.

== Partial comparison
Depending on how the model is implemented, it may also prove useful to perform
a comparison between different parts. For example, if both models rely on some
approximation of optical depth, it may prove useful to compare the results of
that directly, instead of the final result. This can also help in finding where
the two models differ in output.

== Computational costs
Besides visual differences between models, it is also possible to compare them
in terms of computational costs. For memory usage, this should be trivial to
determine, as it is possible to determine the size of any used lookup table,
and calculate the memory usage based on this size.

Comparing execution speed is more difficult, as it depends on the hardware the
model is run on, how it is implemented, and how often it is evaluated. While it
is possible to get an estimate of this using the number of operations that have
to be performed, the real performance has to be measured, using a profiler such
as NVIDIA Nsight and AMD Radeon GPU Profiler.

= Implementation notes
- All model parameters used are from the new bruneton implementation.
- Preetham outputs it's colors in xyY color space, and thus needs to be converted.
  This is done by first converting to the XYZ color space, then to sRGB linear,
  for the D65 reference white. As the model outputs in lumen, while the parameters
  are in Watt, the output is multiplied by $1 / 628$.
- Path tracer is slightly different than bruneton, double check bruneton?
- Replace luminance with radiance, as that is the correct terminology #todo[this]

= Research goals
As seen in @chap:atmospheric-scattering[Chapter], implementations of real-time
atmospheric scattering can be divided in roughly 3 different types: Numerical
integration, where only single scattering is considered. Precomputing, where
multiple scattering is considered as well, and fitted models, which assume the
viewer remains on the ground.

This research aims to implement a new fitted model of atmospheric scattering,
that can support viewing from higher altitudes, as well as space. This will be
done by reusing the work of approximating the optical depth, as done by
@gpu-pro-3 and @gpu-gems-atmo, as well as splitting the model into separate
parts that can be fitted independently, similar to how @bruneton-atmo and
@seb-skyatmo precompute the model in separate parts.

While implementing this, this research aims to answer the following questions:

1. To what extent is it possible to create a fitted model for atmospheric
  scattering with both a viewer from the ground, higher altitude, and space?

If this is possible, it also makes sense to measure if the resulting model is
different in performance characteristics:

2. How big is the performance tradeoff of using this model compared to
  lookup-table based models?

As the fitted models in the discussed previous work are trained for earth's
atmosphere, it may also prove useful to fit it on different atmosphere
parameters than those of earth. Most notably, the density function in most
previous work discussed is $rho(h) = exp(-h / h_0)$. Other planets may have a
different $rho(h)$ that fits better. @bruneton-new and @seb-skyatmo assume
ozone absorption occurs, where the ozone distribution uses a different $rho(h)$.
For this reason, the following question is asked:

3. How difficult is it to adapt this model to use a different $rho(h)$, in
  order to model ozone absorption and other effects?

== Methods
The research can be split into the following tasks:
- Implement a number of the discussed methods, and a reference path tracer.
- Explore ways to create the fitted model.
- Create a fitted model.
- Compare outputs visually.
- Compare outputs performance wise.

=== Implement discussed models
This task requires implementing some of the discussed models in a way that
allows comparing their outputs. This has already partly been done, by
implementing a tool that can run shaders on the GPU, and display the final
result to an `exr` image. From here, it is possible to port existing
implementations to work with this tool. This has already been done for the
model described in @preetham and @bruneton-new. This still needs to be done for
@gpu-pro-3, and @seb-skyatmo for a good overview of the different techniques.
If time permits, more can be implemented for comparison.

@seb-skyatmo provides a path tracer in their provided code, which has been
tested against @PBRT3e for correctness, and can thus be used as the reference
path tracer. This also still needs to be ported to work with the tool.

=== Explore ways to create the fitted model
As there are many ways to create a fitted model, these ways need to be explored.
On the atmosphere side, the focus is on finding a good way to split up the
calculations of atmospheric scattering in a way that may simplify the models
that need to be fit on these results.

There are also many different methods to create a fitted model, such as via
polynomials, used by @gpu-gems-atmo, fitting the coefficients of some formula,
used by @preetham and @hosek-wilkie, or finding another approximation without
fitting, such as @gpu-pro-3. Other methods may also be used, such as training
a neural network, or symbolic regression. This may also require working with
new tools and libraries.

=== Create a fitted model
Using the methods found for creating a fitted model, actually create a fitted
model. If time permits, this may be repeated several times in combination with
evaluating the output and performance of the resulting fitted model, in order
to improve either accuracy compared to the path traced reference, or improve
its performance.

=== Compare model outputs and performance
After a fitted model is created, it can be compared to some of the previously
discussed ones in terms of accuracy to the path traced reference, as well as
runtime performance. Evaluating the previous models by comparing them to the
path tracer only has to be done once. If time permits, multiple fitted models
may end up being tested, causing this step to be repeated for those models.

=== Limitations
To stay within the time constraints and scope of a master thesis, some
limitations should be taken into account. While it may be interesting to
provide an accurate model for earth's atmosphere doing so requires extra work
if the model is to also work for other atmospheres. For this reason it is a
non-goal. This includes detailed spectral rendering.

@bruneton-atmo and @seb-skyatmo provide different methods for dealing with
terrain, such as mountains, shadowing the atmosphere. While interesting, these
methods may require too much time to implement in the given time.

Interactions with other atmosphere effects, such as clouds, aurora, and airglow
is also out of scope.

#pagebreak()

= Planning
#import "@preview/timeliney:0.2.0"
#timeliney.timeline(show-grid: true, {
  import timeliney: *

  headerline(
    group(([Apr], 4)),
    group(([May], 4)),
    group(([Jun], 4)),
    group(([Jul], 4)),
    group(([Aug], 4)),
    group(([Sept], 2)),
  )

  taskgroup(title: [*Implement reference \ models*], {
    task([Path tracer], (0, 1), style: (stroke: 4pt + gray))
    task([@gpu-pro-3's model], (1, 2), style: (stroke: 4pt + gray))
    task([@seb-skyatmo's model], (2, 5), style: (stroke: 4pt + gray))
  })

  taskgroup(title: [*Explore fitting methods*], {
    task([Explore function fitting \ methods], (4, 6), style: (
      stroke: 4pt + gray,
    ))
    task([Find way to split \ atmosphere calculations], (5, 8), style: (
      stroke: 4pt + gray,
    ))
  })

  taskgroup(title: [*Model Fitting*], {
    task([Fit model], (6, 11), style: (stroke: 4pt + gray))
    task([Report results], (9, 12), style: (stroke: 4pt + gray))
  })

  taskgroup(title: [*Compare fitted model \ to previous work*], {
    task([Set up tools], (8, 10), style: (stroke: 4pt + gray))
    task([Compare visual results], (9, 12), style: (stroke: 4pt + gray))
    task([Compare performance \ results], (12, 15), style: (stroke: 4pt + gray))
    task([Report results], (12, 17), style: (stroke: 4pt + gray))
  })

  taskgroup(title: [*Write report*], {
    task([Report results], (16, 19), style: (stroke: 4pt + gray))
    task([Improve report], (18, 20), style: (stroke: 4pt + gray))
    task([Present results], (19, 20), style: (stroke: 4pt + gray))
  })

  milestone(at: 20, style: (stroke: (dash: "dashed")), [*Deadline*])
})

#bibliography("citations.bib", style: "apa", title: "References")
