#let author = "Dimas Leenman"
#let student_num = "0513502"
#let study_prog = "master Computing Science\nUtrecht University"
#let title = "A visual and performance comparison of atmospheric scattering models"
#let supervisor_fst = "Peter Vangorp"
#let supervisor_snd = "Alex Telea"

#let abstract = [
  Previous models of atmospheric scattering have either utilized an analytical
  fitted function, limiting the viewer to the ground, or a number of lookup tables
  improve runtime performance.

  This work introduces 3 new models, one neural network based model that
  supports both ground and space views, one analytical model that does not
  require fitting on reference data, and one model using an approximation of
  the transmittance tables.

  The new models are compared to the existing models, as well as a path traced
  reference, on visual accuracy, runtime performance, and implementation complexity.
]

#set math.equation(numbering: "1.") // allow math equation referencing
#set heading(numbering: "1.") // heading numbering
#set text(font: "New Computer Modern") // nicer font
#set math.cases(gap: 1em) // bit nicer
#let erfc = math.op("erfc") // erfc as math op
#let erf = math.op("erf") // and erf

// reverse F for the correct logo
#show "nvidia-flip": box(stack(dir: ltr, scale(x: -100%, "F"), "LIP"))

// recall something
#let recall(reference) = {
  context {
    // don't take the body, so we can copy the entire figure
    query(reference).first()
  }
}

// appendix
#let appendix(body) = {
  set heading(numbering: "A.1:", supplement: [Appendix])
  // TODO: make the outline better?
  counter(heading).update(0)
  body
}

#import "@preview/cetz:0.4.2" // for diagrams
#import "@preview/lilaq:0.5.0" as lq // for plotting

// document front page setup
#set document(author: author, title: title, description: abstract)

#pad(y: 64pt, align(center, text(24pt)[#title]))
#align(center)[November 1, 2025]

#pad(y: 64pt, align(center, grid(
  columns: (40%, 40%),
  gutter: 24pt,
  [_Author_ \ #author \ (#student_num)], [_First Supervisor_ \ #supervisor_fst],
  [_Study Programme_ \ #study_prog], [_Second Supervisor_ \ #supervisor_snd],
)))

#pagebreak()

#outline()
#pagebreak()

#set page(paper: "a4", numbering: "1") // page numbering
#counter(page).update(1)

= Abstract
#abstract

= Introduction
As games become increasingly more realistic, more accurate visual
effects are needed to render them. In games such as Microsoft Flight
Simulator @fs2024, DCS @DCS, and the Scatterer mod for Kerbal Space
Program @ksp-scatterer the atmosphere is an important element to get
right. Other games, such as Gran Turismo 7 @gt7 make use of a complex
sky simulation to ensure both the sky itself and resulting lighting
looks correct. This work introduces 3 new models of atmospheric
scattering that attempt to improve over existing methods.

#figure(
  grid(
    columns: 3,
    gutter: 3pt,
    image("imgs/msfs.png"), image("imgs/scatterer.png"), image("imgs/gt7.jpg"),
  ),
  caption: [Microsoft Flight Simulator, Scatterer and Gran Turismo 7],
)

= Atmospheric scattering
<chap:atmospheric-scattering>
All particles in the atmosphere of the earth affect how light is
scattered inside it. When a photon hits a particle, it can either be
scattered into a new direction, or absorbed. Which of these events
happen, depends on the wavelength of the photon, as well as the type and
size of the particle. For every photon, these events can happen 0 times,
1 time, referred to as single scattering, or more than once referred to
as multiple scattering.

Before going over previous work, it may prove useful to provide a standard
set of symbols, coordinates, as well as a short
introduction on the physics behind atmospheric scattering.

== Coordinate system
The figure below shows the coordinates used for a viewer $v$ at height $h$
in an atmosphere, with light coming from light source $l$. The coordinate
system for this is described below.

#figure(caption: [Viewer $v$ in an atmosphere.], cetz.canvas({
  import cetz.draw: *
  // planet
  arc-through((-5, -3), (0, 0), (5, -3))

  // planet radius
  line((0, 0), (0, -1.5))
  stroke((dash: "dashed"))
  line((0, -1.5), (0, -3))
  arc-through((-8.5, -2), (0, 3), (8.5, -2)) // atmosphere
  line((0, 1), (0, 3)) // surface normal
  stroke((dash: none))
  content((1em, -0.75), [$r$])

  // viewer
  line((0, 0), (0, 1))
  content((1em, 0.5), [$h$])

  // view ray
  line((0, 1), (4, 3))

  // view angle
  arc((0, 2), start: 90deg, delta: -63.43deg, radius: 1)
  content((1em, 1.5), [$theta_v$])

  // sun ray
  stroke((dash: "dashed"))
  line((0, 1), (-4, 3))
  stroke((dash: none))

  // sun angle
  arc((0, 2), start: 90deg, delta: 63.43deg, radius: 1)
  content((-1em, 1.5), [$theta_l$])

  // sun-view angle
  content((1em, 2.3), [$gamma$])

  // sun
  stroke((dash: "dashed"))
  circle((-4, 3), radius: 0.2)
  content((-4.4, 2.6), [$l$])

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((0, 1), radius: 0.1)
  content((-1em, 0.9), [$v$])

  // ray labels
  content((-2.5, 2), [$p_l$])
  content((2.5, 2), [$p_v$])
}))
<fig:angles>

#figure(
  caption: "Coordinates",
  table(
    columns: 2,
    align: (col, row) => (center, center).at(col),
    inset: 6pt,
    [Radius $r$], [The radius of the planet],
    [Height $h$], [The height viewer $v$ is from the surface],
    [View zenith $theta_v$], [The angle between the surface normal, and view direction],
    [Sun zenith $theta_l$], [The angle between the surface normal, and the light direction],
    [Angle $gamma$], [The angle between the view direction, and light direction],
    stroke: 0.5pt
  ),
)
<tab:coord-sys>

== Physics of scattering
This section provides a basic overview of the physics behind a photon
scattering in an atmosphere. A possible path of a photon scattering from
light source $l$ towards viewer $v$ can be seen below.

#figure(caption: [Viewer $v$ in an atmosphere.], cetz.canvas({
  import cetz.draw: *
  // planet
  arc-through((-5, -1), (0, 0), (5, -1))

  // planet radius
  stroke((dash: "dashed"))
  arc-through((-7.5, 1), (0, 3), (7.5, 1)) // atmosphere
  stroke((dash: none))

  // view to scatter
  line((-3, 1), (1, 1))

  // scatter to sun
  line((1, 1), (5, 4))

  // scatter angle
  arc((0.5, 1), start: 180deg, delta: -145deg, radius: 0.5)
  content((0.75, 1.8), [$theta_s$])

  // scatter to sun, multiple
  stroke((dash: "dashed"))
  line((1, 1), (2, 1))
  line((2, 1), (3, 0))
  line((3, 0), (5, 4))

  // sun
  circle((5, 4), radius: 0.2)
  content((5.4, 3.6), [$l$])

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((-3, 1), radius: 0.1)

  // viewer
  content((-3.3, 0.7), [$v$])

  // path a
  content((-0.75, 0.8), [$a$])

  // path b
  content((3.9, 1), [$b$])
}))

Here, path $a$ represents a path with only one scatter event, where the photon
scatters at an angle $theta_s$. Path $b$ represents a path with multiple
scattering events.

When light travels through the atmosphere, it is attenuated based on the
density of the atmosphere it travels through. Given a density function
$rho(h)$ that depends on height $h$ from the surface, the _optical depth $tau$_
from viewer $v$, with direction $theta_v$ can then be calculated with the
following integral:

$ tau = integral_a^b rho(sqrt(t^2 + h^2 + 2 h t cos theta_v)) d t $
<eq:optical-depth>

Where $a$ and $b$ are the start and end points of the ray respectively.
For $rho(h)$, most other implementations discussed in @chap:previous-work[chapter] use
the exponential function for the density of the medium:

$ rho(h) = exp(-h / h_0) $

Here, $h_0$ is the _scale height_, which determines how fast the density
falls off based on height.

For ozone, @bruneton-new and @seb-skyatmo use the following density:

$ rho(h) = max(0, 1 - (|h - 25|) / 15) $

Where $h$ is in kilometers.

The _transmittance $T$_, which indicates how much light is attenuated, can then
be calculated as follows:

$ T = exp(-beta_a tau) $

Where $beta_a$ is the _absorption coefficient_, which determines how much
of the light is absorbed.

Transmittance $T$ represents how much light is attenuated for a single path
in the atmosphere, but does not take scattering into account. When a photon
scatters, it changes direction. The new direction is not uniform, and depends
on a phase function $F(theta_s)$. The phase function used depends on the
scattering medium. For Mie scattering, which is used to model aerosols,
a common phase function to use is the Cornette-Shanks phase function described
by @cornette-shanks. This phase function is used by @bruneton-atmo,
@production-volume-rendering, @nishita93 and others. It is shown below.

$
  F(theta_s) = 3 / (8pi) ((1-g^2)(1+cos^2 theta_s)) / ((2+g^2)(1+g^2 - 2g cos theta_s)^(3 / 2))
$
<eq:phase-cornette-shanks>

An alternative phase function used by @PBRT3e is the Henyey-Greenstein phase
function, described by @henyey-greenstein can be seen below:

$ F(theta_s) = 1 / (4pi) (1-g^2) / ((1 + g^2 + 2g cos theta_s)^(3 / 2)) $
<eq:phase-henyey-greenstein>

Here, $g$ determines how directional the phase function is, with values near 1
being more directional, and thus photons less likely to significantly change
direction upon scattering. For Rayleigh scattering, $g = 0$ is typically used.
@seb-skyatmo uses an isotropic phase function instead, shown below:

$ F(theta_s) = 1 / (4pi) $

Using the phase function, it is now possible to calculate the transmittance
for the entire path that light can take through the scattering medium.
For a path with a single scattering event, the total radiance $L_"scatter"$
reaching the viewer can be calculated as follows:

$ L_"scatter" = E_"light" F(theta_s) beta_s rho(h_s) T_s T_l $
<eq:scatter-radiance>

Where $beta_s$ is the scattering density coefficient, $h_s$ is the height
at which the scatter event takes place. $T_s$ is the transmittance from the
viewer to the scattering point, $T_l$ is the transmittance from the scattering
point to the light source, and $E_"light"$ is the incident radiance coming
from the light source. Note that when the path between the scattering point
and the light source is obstructed, $E_"light" = 0$.

In order to take multiple media types into account, their transmittance must
be combined. This is done by multiplying all transmittance from all media types
together. $L_"scatter"$ calculates the resulting radiance from scattering for
only one medium type, and thus needs to be calculated for all media types as
well. An example of the total radiance $L_"total"$ due to scattering from
two scattering media $m$ and $r$, with respective phase functions,
transmittance, scattering densities and density functions can be seen below:

$
  L_"total" = E_"light" F^m (theta_s) beta^m_s rho^m (h_s) T^m_s T^r_s T^m_l T^r_l
  + E_"light" F^r (theta_s) beta^r_s rho^r (h_s) T^m_s T^r_s T^m_l T^r_l
$
<eq:scatter-radiance-total>

When calculating the radiance of a path with multiple scattering events,
@eq:scatter-radiance can be applied recursively, where $E_"light"$ is
replaced by another invocation of $L_"scatter"$.

= Path tracing
Path tracing is a brute force method that attempts to calculate all possible
paths a photon can take when scattering in an atmosphere. For this reason,
it is often used as a reference when implementing other models of atmospheric
scattering. @seb-skyatmo, @sky-comparison and @fitted-radiance use a path
tracer as a reference. @gt7, @hosek-wilkie and @preetham instead use it to fit
a model to the result of a path tracer.

The path tracer described here is the implementation provided by
@seb-skyatmo, and was tested against @PBRT3e for correctness.
The implementation is based on @production-volume-rendering.

As it is infeasible to compute all possible photon paths, the final image is
created using Monte-Carlo integration. This is done by averaging many
different, random photon paths into a final color value for the pixel.

== Finding the scattering position
In order to find a point at which a scattering or absorption event occurs,
delta tracking is used @delta-tracking. Delta tracking is an extension of
closed-form tracking @closed-form-tracking, which is used for homogeneous
volumes. Tracking allows importance sampling any homogeneous volume with
exponential attenuation according to Beer's law

$ T = exp(-beta t) $

Where $t$ is the length of the path the photon takes through the volume,
and $beta$ is the volume density.

Tracking can then be used to determine the distance $t'$ that the photon
travels before being scattered, using @eq:tracking-dist, where $xi$ is a
uniform random number where $0 <= xi <= 1$.

$ t' = -ln(1 - xi) / beta $
<eq:tracking-dist>

As the atmosphere is modeled as an exponentially decaying density based on
height from the surface, the tracking method cannot be used there.
Instead, it is modified to take this density into account, which results in
_delta tracking_. When a photon scatters at distance $t'$ along its path, but
the density $rho$ at this point is $0$, no scattering can happen.
This is instead treated as a scattering event where the photon continues
to travel in the same direction. If the density $rho > 0$, delta tracking
only considers a scatter event to happen if @eq:delta-smaller holds, where
$rho_("max")$ is the highest density that can be found in the volume.

$ xi < (beta rho) / (beta rho_("max")) $
<eq:delta-smaller>

If this does not hold, delta tracking is performed again, using the predicted
scattering position derived from $t'$ as starting point. In case of absorption,
the photon path does not scatter any light, and thus no light is added, and no
further scattering events are considered. A minimal example of delta tracking
can be seen below:

#figure(
  caption: "Delta tracking",
  ```glsl
  vec3 delta_tracking(vec3 start, vec3 dir) {
      while (true) {
          float xi = random();
          float zeta = random();
          float t = -ln(1.0 - xi) / rho_max;
          if (zeta < beta * volume_density(start + dir * t) / (beta * rho_max)) {
              // scatter event, return where
              return start + dir * t;
          } else if (length(start + dir * t) > max_distance) {
              // outside of volume, stop
              return start - dir;
          } else {
              start = start + dir * t;
          }
      }
  }
  ```,
)
<code:delta-tracking>

At the found scattering position, it is assumed that both a photon from the
light source scatter here, as well as a photon from another scattering event
in the atmosphere.

In order to combine multiple types of media with different scattering
or absorption properties, $rho_"max"$ becomes the sum of all the $rho$ of
the different types of volumes. Then, when testing for a scattering event,
each of the $rho$ is assigned an interval. If $xi$ is within that interval,
there is a scattering event or absorption event for that medium.

== Light contribution
The contribution of light to this scattering event can be computed in several
ways. It is possible to perform delta tracking towards the light source, and
if no scattering event happens, the light contribution is equal to
$F(theta_s)$. It is also possible to compute transmittance $T$ instead,
calculating the optical depth using numerical integration directly,
using a lookup table, or using ratio tracking, as described by
@residual-ratio-tracking.

Ratio tracking is a modification of delta tracking, where instead of testing
whether there is a scattering event at each predicted scattering event at
distance $t`$, the attenuation $T$, initialized with $1$, is multiplied by
@eq:ratio-tracking-mul

$ 1 - (beta rho(h)) / (beta rho_("max")) $
<eq:ratio-tracking-mul>

The resulting transmittance is then multiplied by $F(theta_s)$, and added to the
final color of this path. A minimal example of ratio tracking is as follows:

#figure(
  caption: "Ratio tracking",
  ```glsl
  float ratio_tracking(vec3 start, vec3 dir) {
      float T = 1.0;
      while (true) {
          float xi = random();
          float t = -ln(1.0 - xi) / rho_max;
          start = start + dir * t;
          if (length(start) > max_distance) {
              // outside of volume, stop
              break;
          }
          T *= 1.0 - beta * volume_density(start) / (beta * rho_max);
      }

      return T;
  }
  ```,
)
<code:ratio-tracking>

Performing these steps once results in single scattering.

== Phase function
In order to perform multiple scattering, these steps are repeated with a new
direction for the photon path, up to $N$ times, where $N$ is the scattering
order. This new direction is chosen by importance sampling the phase function.
For both Rayleigh and Mie scattering, no importance sampling is done. However,
for both the Cornette-Shanks phase function is used, with Rayleigh scattering
using $g = 0$.

It is possible to importance sample a phase function, as @PBRT3e explains for
the Henyey-Greenstein phase function in in _chapter 15.2.3: Sampling phase Functions_.

== Final color
As each path is calculated for a single wavelength of light, it needs to be
repeated for each wavelength. To then eliminate noise in the image, this needs
to be repeated several times as well.
The implementation provided by @seb-skyatmo simply selects the red,
green, or blue color channel at random, by drawing from a uniform random
distribution. Based on the color channel that was selected, different
values for the scattering density $beta$ are selected. The other two color
channels are then set to 0 for this contribution. This is then repeated
many times, and the average of this process is used as final color for the
specified pixel.

= Previous work <chap:previous-work>
A number of methods have been developed for rendering atmospheric scattering
in real time. This section provides an overview of various methods that have
been developed for this purpose, as well as how these methods relate to each
other.

== Single scattering
The work discussed in this section covers implementations of the physics
discussed in @chap:atmospheric-scattering[Chapter], assuming only single
scattering occurs. Most of the focus is put on efficiently evaluating the
optical depth $tau$, from @eq:optical-depth, as using numerical integration
of it results in a large amount of evaluations of $rho(h)$. This can quickly
become prohibitively expensive.

One of the earlier models for atmospheric scattering was developed in
@nishita93, in order to view an atmosphere from space. @gpu-gems-atmo later
shows that this technique can be adapted to work from a ground view as well.

In order to keep the computational cost down, @nishita93 assumes that multiple
scattering can be ignored as it has a negligible effect. They also assume
absorption due to ozone is negligible, and that the density of the air falls
off exponentially, as follows:

$ rho(h) = exp(-h / h_0) $

@nishita93 assumes only two types of scattering can happen. Rayleigh
scattering, for smaller particles, and Mie scattering, for larger particles.
For both types of scattering, they opt to use the Cornette-Shanks phase
function as described in @cornette-shanks:

#recall(<eq:phase-cornette-shanks>)

For Rayleigh scattering, they assume $g = 0$. For Mie scattering, it is
assumed that$0.7 <= g <= 0.85$. The scale heights $h_0$ are set to
$h^r_0 = 7994m$ and $h^m_0 = 1200m$ for Rayleigh and Mie respectively.

In order to reduce computational complexity, @nishita93 precomputes the
optical depth, and stores it in a 2D lookup table. Recall the equation for
the optical depth:

#recall(<eq:optical-depth>)

When assuming that the integral starts at the viewer's position on the view
ray, and ends at infinity, and the air density falls of exponentially, it can
be rewritten as follows:

$ tau = integral_0^infinity exp(sqrt(t^2 + h^2 + 2 h t cos theta_v) / h_0) d t $

As the scale height does not change often, it can be assumed that the integral
now only depends on the height of the viewer $h$, and the view direction
$theta_v$. This allows these parameters to be used as axes in the 2D lookup
table.

For the height, $N$ values of $h_i$ are chosen, with a spacing of

$ h_i = h_0 log((1 - i) / N) + h $

Where $h_i$ is the height used to compute the $i$th row of the lookup table.
Then, the columns of the lookup table, which depend on the viewing angle,
also have to be computed. This is done by taking the heights $h_i$, and
imagining them as spheres around the center of the planet. Then, a number of
cylinders, aligned to the light direction is swept through the spheres.
The index of the sphere, and the index of the cylinder are then used to look
up the row and column of the lookup table.

To calculate the optical depth, trapezoidal integration is used.

To arrive at a final color for the red, green, and blue color channels,
the equation shown in @eq:scatter-radiance-total is evaluated for a number
of points along the view path, and added together.

#recall(<eq:scatter-radiance-total>)

This is then done for every pixel in the output image.

The transmittance for the light direction is calculated using the optical
depth stored in the lookup table. The transmittance for the view direction
is calculated using trapezoidal integration, as this can be performed at the
same time the radiance is accumulated.

From here on, the work of @nishita93 remains used as a base for single
scattering. Most of the variations presented next vary from their
implementation in the manner of how they implement the numerical integration
of the optical depth integral.

@oneil-previous presents a different formulation of the lookup table that
@nishita93 uses. Instead of using the spheres intersecting the cylinders,
it is possible to simply divide the height between $h = 0$ and $h = r_a$, where
$r_a$ is a defined radius of the atmosphere. Then when looking up the value,
@oneil-previous linearly interpolates between the entries in the lookup table
for the specific height. For the view angle, they use the result of
$cos theta_v$ to map to a column in the lookup table.

Besides storing the optical depth for Rayleigh and Mie, @oneil-previous also
stores the density $rho(h)$ for Rayleigh and Mie in the lookup table. If the
view path for the optical depth intersects the planet, this is set to $0$,
which emulates the effect of a planet shadow.

Then, in order to compute the transmittance of the view path segment, the
lookup table is used again, and the segment optical depth is calculated the
same way as @nishita93 does it. However, if the view path intersects the
planet, the view direction the lookup table is sampled with is rotated by
$180degree$, and the order in which they are subtracted is swapped. This
avoids floating point precision errors due to a very large optical depth.

Later, @gpu-gems-atmo modifies their previous work in @oneil-previous to run
on the GPU. Instead of performing the full calculation for each pixel, they
move it to the vertex shader, and let the GPU interpolate between vertices to
get the final per pixel result. The phase functions $F(theta_s)$ are however
calculated per pixel, as this resolves some artifacts.

As GPUs at the time did not have the ability to read from a texture from the
vertex shader, it is not possible to use a lookup table. Instead,
@gpu-gems-atmo finds that it is possible to approximate the optical depth
using a different formula. For the height, they find that $exp(-4h)$ is a
common factor for all view angles.
For the view angles, they find that taking the logarithm of the view angle
results in an unknown curve, and fit a polynomial to this curve.
This results in the following equation:

$ tau = exp (- s_1 h) s_2 exp (5.25x^4 - 6.80x^3 + 3.83x^2 + 0.459x - 0.00287) $

Where $s_1$ and $s_2$ are constants that depend on the atmosphere parameters.
Of note is that this equation for optical depth only works for one combination
of planet radius $r$ and scale height $h_0$. If these are changed, the
polynomial and $s_1$ and $s_2$ have to be fit again. Using this, @gpu-gems-atmo
implements their single scattering the same way as described in
@oneil-previous.

Besides the polynomial approximation @gpu-gems-atmo finds for the optical
depth, there has also been work in physics literature to solve a similar
problem. The result is the Chapman function $"Ch"(x, chi)$, first introduced
in @chapman. As this function cannot be computed exactly, other approximations
have been formulated, such as by @kocifaj, @chapman-approx-big. A more
general overview can be found in @vasylyev-chapman.

@gpu-pro-3 derives a new approximation of the Chapman function, that is
simpler and more numerically stable when implemented using 32 bit floating
point. @gpu-pro-3 first starts with the Chapman function presented in @kocifaj:

$
  "Ch"(x , chi) = 1 / 2 [cos chi + exp (frac(x cos^2 chi, 2)) erfc (sqrt(frac(x cos^2 chi, 2))) (1 / x + 2 - cos^2 chi) sqrt(frac(pi x, 2))]
$
<eq:chap-kocifaj>

Where

$ x = (r + h) / h_0, quad "and" quad chi = theta_v $

The optical depth from viewer $v$ can then be calculated as follows:

$ tau = "Ch"((r + h) / h_0, theta_v)exp(-h / h_0) $
<eq:optical-depth-chapman>

In order to simplify @eq:chap-kocifaj, @gpu-pro-3 makes the following
assumptions:

$
  "Ch"(x, chi) = "Ch"(x, -chi), quad "Ch"(x, 0) = 1, quad "and" quad lim_(x arrow.r oo) "Ch"(x, chi) = 1 / (cos chi)
$

Using this, they find an approximation that works well when $chi < 90 degree$:

$ "Ch"'(c, chi) approx c / (1 + (c - 1) cos chi) quad "if" chi < 90 degree $
<eq:chapman-lt-90>

where $c = "Ch"(x, 90 degree)$. In order to then derive the Chapman function
for $chi > 90 degree$, it is possible to restructure the problem. The optical
depth from the viewer to infinity is the same as the optical depth from
infinity behind the viewer to infinity in front of the viewer. Then, by
subtracting the optical depth from the viewer to infinity behind the viewer,
the optical depth from the viewer to infinity can be calculated. This is
expressed as follows:

$
  "Ch"(x, chi) = 2 exp(x - x sin chi) "Ch"(x, 90degree) - "Ch"(x, 180degree - chi)
$
<eq:chapman-gt-90>

This means, if $"Ch"(x, chi)$ is known for $chi < 90degree$, it is also known
for $chi >= 90degree$. Rewriting @eq:optical-depth-chapman, @eq:chapman-lt-90,
and @eq:chapman-gt-90, as it is desired to calculate the optical depth
without precision errors, results in the following equation for optical depth:

$
  tau_"Ch" = cases(
    display(& exp(-h / h_0) c / (1 + c cos theta_v) & "if" theta_v < 90 degree),
    display(- & exp(-h / h_0) c / (1 - c cos theta_v) + 2 sqrt(x_0) exp(r / h_0 - x_0) quad & "otherwise")
  )
$

Where

$
  c = sqrt((r + h) / h_0), quad "and" quad x_0 = sqrt(1 - cos^2 theta_v)((r + h) / h_0)
$

@gpu-pro-3 then uses $T_"Ch"$ for the optical depth to implement a single
scattering atmosphere, similar to how @nishita93, and @oneil-previous have done
it.

Of note is another approximation to optical depth by @bruneton-atmo. While not
explained is the paper, they implement an approximation of the optical depth
in the original implementation, used for the transmittance when calculating
the radiance of reflected light of the surface of the planet.
Recall the equation for optical depth:

#recall(<eq:optical-depth>)

Here, the distance from the surface at position $t$ is calculated using:

$ h_t = sqrt(t^2 + h^2 + 2h t cos theta_v) $

By assuming that

$ sqrt(1 + u) approx 1 + u / 2 $

This allows rewriting the optical depth integral, with
$rho(h) = exp(-h / h_0)$, and the path going from 0 to maximum distance $d$,
to the following:

$ tau approx integral_0^d exp(-(t^2 / (2 h) + t cos theta_v) / h_0) d t $

The primitive of this integral is as follows:

$
  Tau(t) approx exp((h cos^2 theta_v) / (2 h_0)) erfc((t + h cos theta_v) / sqrt(2 h dot h_0)) sqrt(pi / 2 dot h dot h_0)
$

$erfc(x)$ can be approximated with

$ erfc(x) approx 2 exp(-x^2) / (2.3192x + sqrt(1.52x^2 + 4)) quad "if" x >= 0 $

When $x < 0$, $erfc(-x) = 2 - erfc(x)$ can be used.
Combining this allows the optical depth to be evaluated from the viewer up to
distance $d$ with the following code:

#figure(
  ```glsl
  float optical_depth(float h_0, float h, float cos_theta_v, float d) {
      float a = sqrt((0.5 / h_0) * h);
      vec2 a01 = a * vec2(cos_theta_v, cos_theta_v + d / h);
      vec2 a01s = sign(a01);
      vec2 a01sq = a01 * a01;
      float x = a01s.y > a01s.x ? exp(a01sq.x) : 0.0;
      vec2 y = a01s / (2.3193 * abs(a01) + sqrt(1.52 * a01sq + 4.0))
          * vec2(1.0, exp(-d / h_0 * (d / (2.0 * r) + cos_theta_v)));
      return sqrt((6.2831 * h_0) * h) * exp((r - h) / h_0)
          * (x + dot(y, vec2(1.0, -1.0)));
  }
  ```,
  caption: [@bruneton-atmo's optical depth approximation],
)

/* original code:
#figure(
    ```glsl
    // optical depth for ray (r,mu) of length d, using analytic formula
    // (mu=cos(view zenith angle)), intersections with ground ignored
    // H=height scale of exponential density function
    float opticalDepth(float H, float r, float mu, float d) {
        float a = sqrt((0.5/H)*r);
        vec2 a01 = a*vec2(mu, mu + d / r);
        vec2 a01s = sign(a01);
        vec2 a01sq = a01*a01;
        float x = a01s.y > a01s.x ? exp(a01sq.x) : 0.0;
        vec2 y = a01s / (2.3193*abs(a01) + sqrt(1.52*a01sq + 4.0))
            * vec2(1.0, exp(-d/H*(d/(2.0*r)+mu)));
        return sqrt((6.2831*H)*r) * exp((Rg-r)/H)
            * (x + dot(y, vec2(1.0, -1.0)));
    }
    ```,
    caption: [Optical depth approximation]
)
<code:bruneton-optical-depth>

The email:
opticalDepth(float H, float r, float mu, float d) computes

D = integral from 0 to d of exp(-(r(t)-Rg)/H) dt

where r(t) = sqrt(r^2+2.t.r.mu+t^2), which, using sqrt(1+u) ~= 1+u/2 for
u small, can be approximated with r(t) ~= r + t.mu + t^2/(2.r).

Replacing this in D gives

D ~= exp((Rg-r)/H) * integral from 0 to d of exp(-(t.mu+t^2/(2r))/H) dt

A primitive of the integrand is (see Wolfram Alpha)

sqrt(pi.H.r/2).exp(mu^2.r/(2.H)).erfc((mu.r+t)/sqrt(2.H.r)) + cste

and

erfc(x) ~= 2.exp(-x^2)/(2.3192*x+sqrt(1.52*x^2+4)) for x >= 0

Using also erfc(-x)=2-erfc(x), this gives the formula used in this function.

*/

== Multiple scattering
Besides single scattering, multiple scattering adds a non-negligible
contribution to the total radiance of the scattered light. On earth, this is
best noticed during sunset and sunrise, as it adds an extra blue glow to the
atmosphere.

When assuming single scattering, the chosen scattering direction is trivial,
as this direction changes from the light directly towards the viewer when a
photon is scattered. When assuming two scattering events per photon path,
this direction is less trivial. The second scatter, where the photon scatters
towards the viewer, now has to integrate over all scattering directions to
account for all the incoming radiance due to scattering.

When using numerical integration, this results in having to evaluate many
single scattering paths for one multiple scattering event. This becomes
prohibitively expensive to compute. In order to still render an atmosphere in
real time, @bruneton-atmo proposes using several lookup tables.

In order to compute the transmittance, @bruneton-atmo uses a lookup table.
Instead of storing the optical depth, and using it to calculate the
transmittance, they store the transmittance directly in the table.
This allows storing the combined transmittance of all scattering media in a single
table, instead of one table for the optical depth of each media type.

Instead of evaluating the single scattering equation directly, @bruneton-atmo
recognizes it can be precomputed as well. Recall the single scattering
equation:

#recall(<eq:scatter-radiance>)

Here, $T_s$ depends on the viewer height $h$ and view direction $theta_v$.
$T_l$ depends on the light direction $theta_l$. The height of the scattering
event can be calculated from $h$ and $theta_v$, as many scattering events
happen along one view ray. The last dependency is the scattering angle $theta_s$.
This means single scattering only relies on 4 different variables, and can thus
be precomputed. The results are stored in a 4D lookup table.

As 4D textures are not available on the GPU, @bruneton-atmo instead chooses to
use a 3D texture, and for each slice of the texture, divide it into multiple
tiles to emulate a 4D texture.

Then, when $L_"scatter"$ is needed, it can simply be retrieved from the lookup
table. Rayleigh and Mie scattering are stored in separate tables.

The contribution of multiple scattering can be precomputed in a similar manner
to how it is done for single scattering. Instead of calculating the incoming
radiance from the sun due to transmittance, each sample point along the view
path integrates the result of single scattering over each direction. For more
than two scatter events, the previous result from the multiple scattering
table is read instead.

Integrating the scattering for all directions on all sample points along the
view path is expensive. To resolve this, @bruneton-atmo uses another lookup
table, that calculates the incoming radiance from multiple scattering for
every scattering direction $theta_s$, viewer height $h$ and light direction
$theta_l$. This lookup table can then be used to evaluate the incoming
radiance due to multiple scattering at each sample point.

This process is then repeated for several steps, in order to compute scattering
for more than 2 scatter events per path, referred to as _scattering orders_ by
@bruneton-atmo. This then results in a final lookup table that incorporates
both the single scattering and multiple scattering. This table is then read to
determine the radiance.

@elek presents a variation on the model presented by @bruneton-atmo. Instead of
using a table with 4 dimensions, @elek chooses to not incorporate the scattering
angle $theta_s$, and use a 3-dimensional table instead. This means that the
phase function has to be calculated, instead of reading it from the table when
computing radiance. It also means the planet shadow cannot be stored in the
table either. @elek argues this effect is negligible enough for this tradeoff
to work.

@seb-skyatmo proposes a different method of precomputing the
scattering. Instead of precomputing single scattering for all viewer positions,
they use a 2D lookup table, and recompute it when either light direction
$theta_l$ or viewer height $h$ changes.

For multiple scattering, @seb-skyatmo chooses to instead approximate
the result, instead of fully calculating each scattering order. To make the
approximation work, @seb-skyatmo assumes that after single scattering,
the phase function becomes isotropic. This eliminates the dependence on the
scattering direction $theta_s$ for multiple scattering.

Second order scattering can then be calculated by, for height $h$, and view
direction $theta_v$, calculate the incoming radiance from scattering for every
direction.

Scattering orders higher than 2 can be approximated by multiplying the second
order scattering by $F_"ms"$, which is a geometric series infinite sum:

$ F_"ms" = 1 + f_"ms" + f^2_"ms" + f^3_"ms" + ... = 1 / (1 - f_"ms") $

$f_"ms"$ represents the radiance that the sample point would receive, if the
atmosphere would emit light, with no other scattering contribution. $f_"ms"$
can thus be calculated by integrating the transmittance over all directions.

== Fitted models
The previously discussed methods have all tried computing the scattering
equations of @chap:atmospheric-scattering[Chapter] directly. This section
introduces a different method, that attempts to find an equation that can be
fitted to the output of a path tracer, in order to then render the atmosphere
without requiring the scattering equations, or precomputed lookup tables.
To keep complexity of the fitted model low, these do assume that the viewer is
either on the ground, or close to it, and thus do not support viewing from
space.

One of the earlier models to do this was presented by @preetham. Their model
is based on an earlier sky luminance model by @perez-formula:

$
  cal(F)(theta, gamma) = (1 + A exp(B / (cos theta)))(1 + C exp(D gamma) + E cos^2 gamma)
$

Here, $A$, $B$, $C$, $D$, $E$ are coefficients that are fitted to match a
reference. The presented model does not directly give radiance for a specific
wavelength, or red, green, and blue channels. Instead, the resulting color
space is _CIE $x y Y$_. This can then be converted to the radiance for a
specific wavelength as @preetham describes.

Radiance $Y$ is calculated as follows:

$ Y = Y_z (cal(F)(theta_v, gamma)) / (cal(F)(0, theta_l)) $

The chromaticity values $x$ and $y$ are calculated similarly:

$
  x = x_z (cal(F)(theta_v, gamma)) / (cal(F)(0, theta_l)) quad "and" quad
  y = y_z (cal(F)(theta_v, gamma)) / (cal(F)(0, theta_l))
$

Details for calculating $Y_z$, $x_z$ and $y_z$ are further described in
@preetham.

The coefficients of the model given in @preetham are calculated by fitting them
to the results of a path tracer. The result is a single formula that can model
the color of the atmosphere, given $h = 0$.

@hosek-wilkie improves upon the findings of @preetham, by taking the reflection
of light from the ground into account. They also improves the aureole,
which is a bright halo around the sun. This halo is also darkened when the sun
is near the horizon. This results in a new formula:

$
  & bb(F)(theta, gamma) =                                                      \
  & (1 + A exp(B / (0.01 + cos theta)))
    (C + D exp(gamma E) + F cos^2 gamma + G chi(H, gamma) + I sqrt(cos theta)) \
$

Where $A$, $B$, $C$, $D$, $E$, $F$, $G$, $H$, and $I$ are the coefficients to
fit. $chi$ is calculated as follows:

$ chi(g, alpha) = (1 + cos^2 alpha) / (1 + g^2 - 2g cos alpha) $

The final radiance for wavelength, or color channel $lambda$ is then
calculated with:

$ L_lambda = bb(F)(theta_v, gamma) L_(M lambda) $

All coefficients are then again fit on the result of a path tracer.
Unlike the work of @preetham, the coefficients of $bb(F)(theta, gamma)$ are
interpolated from a larger set of coefficients, based on turbidity, ground
albedo, and sun elevation $theta_s$.

While these models fit the general sky quite well, they are not capable of
representing more finer details. To address this, @fitted-radiance introduces
a new model based on Canonical Polyadic Decomposition @tensor-decompose. This
method is similar to singular value decomposition, and acts like an image
compression technique. Besides outputting the direct radiance, it also outputs
the transmittance, and polarization of light. Contrary to @preetham and
@hosek-wilkie, this model is also able to represent the sky when the sun is
below the horizon, as well as $h > 0$, up to a maximum height. The downside is
that this requires significantly more coefficients, with the largest version of
the model being 2.2 GB in size.

As @bruneton-atmo and @seb-skyatmo have done with precomputing single and
multiple scattering, it is also possible to store the results of a path tracer
in a lookup table directly, instead of fitting a formula to the results.
This is the method @gt7 takes. For the lookup table, they use 64 2D lookup
tables, each table representing a single time of day, of angle $theta_l$,
separated in $theta_v$ and $gamma$. In order to fix artifacts near the horizon,
more tables are allocated when the sun is close to the horizon. Per table,
a special spherical mapping is used that increases the number of samples near
the sun.

= Research goals <chap:research-goals>
As seen in @chap:previous-work[Chapter], the discussed models of atmospheric
scattering can be divided in roughly 3 categories:
- Numerical integration, where all equations are directly integrated.
  For real-time use, only single scattering
  is considered.
- Precomputed models, where a large part of the integration is done ahead of
  time, allowing for the resulting model to be used in real-time. As there is
  no constraint on the runtime of the precomputation step, these models can
  take multiple scattering into account.
- Fitted models and approximations, where either part of the calculations
  or the entire model is replaced using an approximation, or a fitted model.
  These models make a tradeoff in that they cannot be used in certain circumstances,
  for example, @preetham and @hosek-wilkie cannot be used for viewers that are
  viewing the atmosphere from either higher up, or from space.

From these categories it becomes clear that no general fitted model, that can
be used for both ground and space views, exists. This work aims to create
such a model, and compare it to the previously discussed models.
While doing this, this work attempts to answer the following questions:
- To what extent is it possible to create a fitted model for atmospheric scattering
  that can support both a ground-based viewer, a viewer from higher altitudes, and space?
- How is the runtime performance of this model compared to lookup-table based models?
- How difficult is it to adapt the model to use a different $rho(h)$ than the exponentially
  decaying density the discussed models assume?

= Assumptions
Before making the new model, a number of requirements have to be set for what it
should be capable of doing. The first requirement is that it has to be able to
produce a sky view, as all discussed models in @chap:previous-work[Chapter] 3 can
do so. In order to render sky and space views correctly, both transmittance from
the viewer to the surface and scattering from the viewer to the surface need to
be provided, in order to attenuate light coming from surfaces inside the atmosphere.
@bruneton-atmo provides a method to calculate transmittance and scattering for a
segment of the view path even if there is no way to provide this.

As @bruneton-atmo and @elek calculate multiple scattering as adding on top of
single scattering, it may be possible to calculate single scattering and multiple
scattering separately.

== Model parts
This means that the model needs to provide at least the following:
- Transmittance from viewer to infinity
- Scattering from viewer to infinity.
- Either combined scattering or separate multiple scattering.

Note that this does not include optical depth. As transmittance is derived
from the optical depth, the optical depth approximations provided by @chapman-approx-big,
@vasylyev-chapman, @kocifaj and @gpu-pro-3 can still be used here.

As shown in @bruneton-atmo, transmittance and scattering from the viewer to
infinity is enough to reconstruct both transmittance and scattering along a
segment in the atmosphere. For this reason, it does not have to be modeled directly.

== Coordinate space
The coordinate space used in @chap:previous-work[chapter] uses viewer height $h$, and
viewer direction $theta_v$ to calculate the height along distance $t$ of the
view path. This is illustrated below:

#figure(caption: [Viewer $v$ in an atmosphere.], cetz.canvas({
  import cetz.draw: *
  // planet
  arc-through((-5, -3), (0, 0), (5, -3))

  // planet radius
  line((0, 0), (0, -1.5))
  stroke((dash: "dashed"))
  line((0, -1.5), (0, -3))
  arc-through((-8.5, -2), (0, 3), (8.5, -2)) // atmosphere
  line((0, 1), (0, 3)) // surface normal
  stroke((dash: none))
  content((1em, -0.75), [$r$])

  // viewer
  line((0, 0), (0, 1))
  content((1em, 0.5), [$h$])

  // view ray
  set-style(mark: (end: ">", fill: black))
  line((0, 1), (3, 1))
  set-style(mark: none)

  // view angle
  arc((0, 2), start: 90deg, delta: -90deg, radius: 1)
  content((1em, 1.5), [$theta_v$])

  // view ray, rest
  stroke((dash: "dashed"))
  line((3, 1), (8, 1))

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((0, 1), radius: 0.1)
  content((-1em, 0.9), [$v$])

  // ray labels
  content((2.8, 1.5), [$t$])
}))

The height above the planet surface, along the ray at position $t$,
$h(t)$, can then be calculated as follows:

$ h(t) = sqrt(t^2 + (h + r)^2 + 2 t (h + r) cos theta_v) - r $
<eq:height-along-ray>

For the light direction, $theta_l$ can be used, as described in @chap:previous-work.
Only the view and light angles $theta_v$ and $theta_l$ do not cover all possible
angles that can be used in the phase function. For this reason, $gamma$ is used.
This is the same as described earlier in @fig:angles[figure]:

#recall(<fig:angles>)

= Ground truth
The ground truth rendering is provided by a volumetric path tracer, implemented
as described in @PBRT3e and @production-volume-rendering. To match the other
models described in @chap:previous-work[chapter], the path tracer uses the same
atmospheric density as described in @bruneton-atmo. This means Rayleigh and Mie
scattering are represented using exponentially decaying densities, depending on
altitude, and Ozone density is represented with a tent function.

As correct spectral rendering is not a focus, the path tracer outputs colors
in linear sRGB color space.

= Automatic model finding
As manually finding a formula that fits either the transmittance or scattering
equations may be very time consuming, as well as complex in case of the scattering
equations, it is preferred to find an automated solution for this. For this, it is
useful to have some reference data to work with.

== Reference data
As a ground truth path tracer is available, it is possible to utilize it to obtain
the correct value for transmittance for each $(h, theta_v)$ pair, as well as the
scattering radiance for each $(h, theta_v, theta_l, gamma)$ pair.

The transmittance data is rendered out to a 2D texture. The x axis represents the
height $h$, and the y axis is the view angle $cos theta_v$. The result is shown
below in @trans-table.

For scattering, there are 4 relevant parameters, which are rendered out to a tiled
2D texture. Here, inside each tile, the x, and y coordinates represent the squared
height $h^2$, and view angle $cos theta_v$. The tile x and y coordinates represent
the light angle $cos theta_l$, and the phase function angle $cos gamma$.

For scattering, a 4D texture is used, but it is rendered as a 2D texture, divided
in tiles. Inside each tile, the x axis represents the squared view height $h^2$.
The square gives more samples to the lower altitudes, improving results. The y axis
then represents the view angle $theta_v$.

#grid(
  columns: (auto, auto),
  gutter: 12pt,
  [#figure(
      image("imgs/pathtrace-transmittance.png"),
      caption: [Transmittance data, as image.
        x axis is the viewer height from the surface,
        y axis is the viewing angle $cos theta_v$],
    ) <trans-table>],
  [#figure(
      image("imgs/pathtrace-scattering-proper.png"),
      caption: [Scattering data, as image.
        - x axis is the light angle $cos theta_l$,
        - y axis is the angle between the light and view rays, $cos gamma$.
        - Inside each tile, the x axis is the squared viewer height from the surface,
        - The y axis is the viewing angle $cos theta_v$
      ],
    ) <scatter-table>],
)

As there is no scattering when the sun is obscured by the planet, the light angle
$theta_l$ is limited to be between $0 degree$ and $100 degree$. As the phase
function angle $gamma$ is clamped to come from the given combination of $theta_v$
and $theta_l$.

For both the transmittance, and scattering, the tables parameterized according to
@bruneton-atmo were also generated, but these didn't give significantly better or
worse results when trying to fit functions to these tables.

== Observations
When looking at the transmittance data, it becomes clear the transmittance per
color channel follows a curve similar to the one defined by the formula $exp(-exp(-f(h, theta_v)))$,
where $f(x)$ is some unknown function based on the viewer height $h$ and angle $theta_v$.
When attempting to fit manually, $ f(h, theta_v) = h^2 [-1 + cos theta_v] $ appears to
be a good approximation, but only for low values of $h$. As $exp(-exp(-x))$ has the general
shape of the sigmoid function: $ 1 / (1 + exp(-x)) $

== symbolic regression
One method of automatically finding a function that fits given data is symbolic regression.
`PySR` @pysr is a python library that can perform symbolic regression, and claims to better
than some other methods and libraries available at finding functions.
It however was not capable of finding a suitable function that fit the transmittance data.
Even on simplified data, with only one exponentially decaying media, and no ozone
layer, it was only able to find the $ exp(-exp(-f(h, theta_v))) $ formula, however,
the found formulas for this did not function beyond specific ranges.

As symbolic regression worked badly for transmittance, it was not attempted for scattering,
as the scattering data is more complex.

== Kolmogorov-Arnold networks
Another method that can potentially produce a function that fits the given data are
Kolmogorov-Arnold networks @pykan. Due to limitations in how Kolmogorov-Arnold networks
function, they have difficulty representing the steep cutoff needed for the transmittance
data. For this reason, they did not fit the data well, and thus weren't tried for scattering.

== Polynomial fit
As a polynomial has trouble representing the steep cutoff, it has to be used in
conjunction with another method that can. For this reason, the result of the
polynomial is put into the sigmoid function. A separate polynomial is used
for the height $h$ and $theta_v$, which are then added together. From experiments
it becomes clear that the height does not need more than one degree for the
approximation to work. For $theta_v$, only a second degree polynomial is needed.
This results in the following formula:

$ 1 / (1 + exp(-a_1 - a_2 h - a_3 cos theta_v - a_4 cos^2 theta_v)) $

Where $a_n$ are the parameters used. This is then done once for Mie, and once for
Rayleigh.

== Neural networks <chap:neural-networks>
The last automatic method tried are neural networks, specifically multilayer
perceptrons, using the python library pytorch @pytorch. For activation functions,
both the sigmoid and the rectified linear unit were tried. For training, the Adam
optimizer @adam was used, as other optimizers did not converge. A learning rate
of $0.01$ was used.

=== Transmittance
For the transmittance data, a neural network with a single hidden layer of size 4,
with the sigmoid activation function was trained. More layers, or layers with more
neurons did not significantly improve results. The neural network converged for both
the direct transmittance table and the parameterization described in @bruneton-atmo.

=== Scattering
For scattering, a neural network with two hidden layers, of size 8 was trained.
For the activation function of the final layer, $exp(-x)$ is used.
Reducing the amount of hidden layers or using smaller hidden layers made the
resulting approximation of the scattering data visibly worse. Adding more layers
or increasing the layer size did not improve results much.

=== Implementation as atmosphere
As the `GLSL` shading language supports up to `4x4` matrices, implementing the
neural network for transmittance is trivial, as the weights for the network
can be directly represented using the `mat2x4`, `mat4x4` and `mat4x3` types.
The biases can be represented using the `vec4` and `vec3` types. This allows
the neural network to be directly implemented into `GLSL`.

As the scattering neural network uses a hidden layer size of 8, it is no longer
possible to represent directly in the types available in `GLSL`. Instead, the matrices
used for the weights in the hidden layers have to be split up. The hidden layer
states also have to be split up into two `vec4`'s.

As the neural network is only trained for viewer positions inside the atmosphere,
any viewer that is outside the atmosphere is moved to the top atmosphere boundary.

In order to calculate the transmittance for a given segment, the same method
as described in @bruneton-atmo is used. The neural network is trained on the
transmittance of the full ray, and if the ray intersects the surface, the transmittance
at this intersection is calculated using the neural network. The full transmittance is then
calculated as follows:

$ T = T_"viewer" / T_"intersect" $
where $T_"viewer"$ is the transmittance from the viewer, and $T_"intersect"$ is the transmittance,
in the view direction, at the point the view ray intersects the planet.

As the neural network is not accurate enough when the view ray is looking towards
the planet surface, the rays are instead reversed, and the intersection position
is used as the full ray transmittance. The viewer position is then used as the
intersect position.

Scattering is again done using the method described in @bruneton-atmo. The neural
network is used to calculate scattering for the full ray. If it hits the planet surface,
the scattering from that position onward is then removed from the final result, as follows:

$ S = S_"viewer" - S_"intersect" (T_"intersect" / T_"viewer") $

Note that the transmittance of the viewer and intersect point look in the opposite direction
as the view ray, as this improves accuracy, as explained earlier.

= Manual modeling
Besides of using automatic methods to find an approximation to the scattering equations,
a manual approach may also provide a useful approximation.

== Flat homogeneous atmosphere <chap:flat-atmo>
The simplest possible model for scattering is a flat homogeneous atmosphere, with
the viewer $v$ being at a distance of $h_v$ below the top of the atmosphere boundary.
Assuming only one scattering medium type, as well as both the viewer and light
source $l$ above the horizon, meaning $0 degree <= theta_v <= 90 degree$ and
$0 degree <= theta_l <= 90 degree$. This looks as follows:

#figure(caption: [Viewer $v$ in a flat atmosphere.], cetz.canvas({
  import cetz.draw: *
  // planet
  line((-8, 0), (8, 0))

  // atmosphere
  stroke((dash: "dashed"))
  line((-8, 3), (8, 3))

  // sun ray
  line((0, 1), (-5, 3))

  // sun
  circle((-6.5, 4), radius: 0.2)
  content((-6.1, 3.8), [$l$])
  stroke((dash: none))

  // viewer
  line((0, 3), (0, 1))
  content((1em, 2.5), [$h_v$])

  // view ray
  line((0, 1), (5, 3))

  // view angle
  arc((0, 2), start: 90deg, delta: -68deg, radius: 1)
  content((1em, 1.55), [$theta_v$])

  // sun angle
  arc((0, 2), start: 90deg, delta: 68deg, radius: 1)
  content((-1em, 1.55), [$theta_l$])

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((0, 1), radius: 0.1)
  content((-1em, 0.9), [$v$])
}))

To then calculate single scattering, the following integral has to be solved,
where $beta_a$ is the absorption coefficient of the atmosphere, and $beta_s$
the scattering coefficient.

$
  L_"scatter" = integral_0^h_v beta_s
  exp(-t beta_a / (cos theta_v)) exp(-(h_v - t) beta_a / (cos theta_l)) dif t
$ <eq:flat-atmo-integral>

To make integration easier, this can then be rewritten as follows:

$
  L_"scatter" = beta_s h_v
  integral_0^1 exp(-t beta_a / (cos theta_v) - (1 - t) beta_a / (cos theta_l)) dif t
$

As this can be integrated exactly, this then becomes:

$
  L_"scatter" = beta_s h_v (exp(-beta_a / (cos theta_v)) - exp(-beta_a / (cos theta_l))) / (beta_a / (cos theta_l) - beta_a / (cos theta_v)) dif t
$

The shader presented in @cheap-sky utilizes this to then approximate the single
scattering equations in a spherical atmosphere.

== Spherical atmosphere
When the planet radius is large enough, with a small enough radius for the atmosphere,
using a flat atmosphere can serve as a good enough approximation. @cheap-sky does this,
and replaces $beta_a / (cos theta_v)$ and $beta_a / (cos theta_l)$ with the optical depth
of a homogeneous sphere. Under the assumption that the sphere extends up to the scale
height of an exponentially decaying atmosphere, with the viewer standing on the
planet, at distance $r$ away from the planet center. As the atmosphere remains homogeneous,
the optical depth is the distance from the viewer to the edge of the atmosphere. The
optical depth can thus be expressed as a line starting from distance $r$ from the
origin, intersecting a circle of radius $r + h_0$:

// alternative: $ beta_a * sqrt(h_s * (h_s + 2 * r) + r^2 * cos^2 theta_v) - beta_a * r * cos theta_v $
$
  tau_"sphere" = beta_a sqrt((r + h_0)^2 - r^2 + r^2 cos^2 theta_v)
  - beta_a r cos theta_v
$

When the difference between $r + h_0$ and $r$ is set equal to one, this can then be
rewritten to the _DR-1_ (shown below) form described in equation 9 in @overview-chap:

$ f(theta_v, a) = sqrt(1 + 2 a + a^2 cos^2 theta_v) - a cos theta_v $
<eq:dr-1-form>

Where $a$ is the radius of the sphere, relative to the scale height $h_0$.

When plotted against the optical depth integral, with $rho(h) = exp(-h / h_0)$,
it shows this approximation is reasonable for any view angle above the horizon.
Shown below is the approximation compared to the reference integral, as well as
the approximation described in @gpu-pro-3, for a planet with a radius of 600, and
a scale height of 1. $tau$ is the optical depth, and $theta$ is the angle of the
view ray with the surface normal of the planet.

#figure(
  {
    // angle to go over
    let domain = lq.linspace(0, 1)

    // planet radius / scale height
    let r = 600

    // integral
    let p1 = x => {
      let tot = 0
      let n = 0
      let t = 0
      let dt = 1 / 2
      while n < 256 {
        tot += calc.exp(r - calc.sqrt(t * t + r * r + 2 * t * r * x)) * dt
        t += dt
        n += 1
      }
      tot
    }

    // DR-1
    let p2 = x => {
      calc.sqrt((r + 1) * (r + 1) - r * r + r * r * x * x) - r * x
    }

    // schuler
    let p3 = x => {
      let c = calc.sqrt(calc.pi * r / 2)
      c / (c * x + 1)
    }

    lq.diagram(
      width: 100%,
      height: 192pt,
      xaxis: (label: [$cos theta$]),
      yaxis: (label: [$tau$]),
      // integral
      lq.plot(
        domain,
        p1,
        label: "Integral",
        mark: none,
        smooth: true,
        color: black,
        stroke: (thickness: 1pt),
      ),

      // DR-1
      lq.plot(
        domain,
        p2,
        label: "Approximation",
        mark: none,
        smooth: true,
        color: black,
        stroke: (thickness: 1pt, dash: "dashed"),
      ),

      // schuler
      lq.plot(
        domain,
        p3,
        label: "Schuler",
        mark: none,
        smooth: true,
        color: black,
        stroke: (thickness: 1pt, dash: "dotted"),
      ),
    )
  },
  caption: [Scaled optical depth],
) <fig:optical-depth-plots>

Note that this approximation only works when the viewer is looking up, with
$0 degree <= theta_v <= 90 degree$.

@overview-chap show that this approximation can also be applied to other optical
depth profiles besides the exponential one.

From here, @cheap-sky uses this to calculate the total amount of single scattering
for a path in the atmosphere. With the following optical depths:

$ tau = beta_a sqrt((r + h_0)^2 - r^2 + r^2 cos^2 theta) - beta_a r cos theta $
<eq:optical-depth-sphere>

Where $theta$ is $theta_v$ for a view ray, and $theta_l$ for a light ray.
As this approximates the normalized optical depth for a given altitude, calculating
the optical depth at a different altitude $h$ means simply multiplying the optical depth
by the medium density at the viewer altitude:

$
  tau = beta_a exp(-h / h_0) (sqrt((r + h + h_0)^2 - (r + h)^2 + (r + h)^2 cos^2 theta) - (r + h) cos theta)
$

The total single scattering can now be calculated as follows, using the scattering
integral for flat homogeneous atmospheres:

$
  L_"scatter" = tau_"view" beta_s F(theta_s)
  ((exp(beta_a tau_"view") - exp(beta_a tau_"light")) / (beta_a tau_"light" - beta_a tau_"view"))
$ <eq:scatter-flat>

This provides a reasonably accurate approximation when the view ray is looking up.

== Looking down
The optical depth approximation however does not work well when looking down, below
the horizon. The initial approximation provided by @gpu-pro-3 has the same issue.
Consider a viewer in the atmosphere looking in a direction where $theta_v = 90 degree$:

#figure(
  caption: [Viewer $v$ in an atmosphere, looking at the horizon],
  cetz.canvas({
    import cetz.draw: *
    // planet
    arc-through((-5, -3), (0, 0), (5, -3))

    // planet radius
    line((0, 0), (0, -1.5))
    stroke((dash: "dashed"))
    line((0, -1.5), (0, -3))
    arc-through((-8.5, -2), (0, 3), (8.5, -2)) // atmosphere
    line((0, 1), (0, 3)) // surface normal
    stroke((dash: none))
    content((1em, -0.75), [$r$])

    // viewer
    line((0, 0), (0, 1))
    content((1em, 0.5), [$h$])

    // view ray
    line((0, 1), (8, 1))

    // view angle
    arc((0, 2), start: 90deg, delta: -90deg, radius: 1)
    content((1em, 1.5), [$90 degree$])

    // viewer point
    stroke((dash: none))
    fill(black)
    circle((0, 1), radius: 0.1)
    content((-1em, 0.9), [$v$])

    content((2.5, 1.5), [$p_v$])
  }),
)

The optical depth of this view path $p_v$ is equal to half the optical depth of
a viewer at an infinite distance away from the atmosphere, looking through it.
This total optical depth can then be calculated as $2 tau$, as seen in the view
ray $p_t$:

#figure(
  caption: [View ray for a viewer at an infinite distance away from the atmosphere],
  cetz.canvas({
    import cetz.draw: *
    // planet
    arc-through((-5, -3), (0, 0), (5, -3))

    // planet radius
    line((0, 0), (0, -1.5))
    stroke((dash: "dashed"))
    line((0, -1.5), (0, -3))
    arc-through((-8.5, -2), (0, 3), (8.5, -2)) // atmosphere
    line((0, 1), (0, 3)) // surface normal
    stroke((dash: none))
    content((1em, -0.75), [$r$])

    // viewer
    line((0, 0), (0, 1))
    content((1em, 0.5), [$h$])

    // view ray
    line((0, 1), (8, 1))
    line((0, 1), (-8, 1))

    // view angle
    arc((0, 2), start: 90deg, delta: -90deg, radius: 1)
    arc((0, 2), start: 90deg, delta: 90deg, radius: 1)
    content((1em, 1.5), [$90 degree$])
    content((-1em, 1.5), [$90 degree$])

    content((2.5, 1.5), [$p_t$])
  }),
)

Now consider the viewer looking down at the atmosphere. In that case, the
optical depth becomes the optical depth for the full view ray, from the closest
point to the planet $c$, without the optical depth of the part of the ray $p_h$
behind the viewer, as seen below:

#figure(caption: [Viewer $v$ looking down], cetz.canvas({
  import cetz.draw: *
  // planet
  arc-through((-5, -3), (0, 0), (5, -3))

  // planet radius
  line((0, 0), (0, -1.5))
  stroke((dash: "dashed"))
  line((0, -1.5), (0, -3))
  arc-through((-8.5, -2), (0, 3), (8.5, -2)) // atmosphere
  line((0, 1), (0, 3)) // surface normal
  stroke((dash: none))
  content((1em, -0.75), [$r$])

  // viewer
  line((0, 0), (0, 1))
  content((1em, 0.5), [$h$])

  // view ray
  line((0, 1), (8, 1))
  line((0, 1), (-6, 1))
  stroke((dash: "dashed"))
  line((-6, 1), (-8, 1))
  stroke((dash: none))

  // viewer point
  line((-6, 1), (-3.5, -1.5))
  stroke((dash: "dashed"))
  line((-6, 1), (-7, 2))
  line((-2.5, -2.5), (-3.5, -1.5))
  stroke((dash: none))

  // view angle
  arc((0, 2), start: 90deg, delta: -90deg, radius: 1)
  arc((0, 2), start: 90deg, delta: 90deg, radius: 1)
  content((1em, 1.5), [$90 degree$])
  content((-1em, 1.5), [$90 degree$])

  // viewer angle, at v this time
  arc((-7, 1), start: 180deg, delta: -45deg, radius: 1)
  content((-6, 1.6), [$45 degree$])

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((-6, 1), radius: 0.1)
  content((-6, 0.5), [$v$])

  circle((0, 1), radius: 0.1)
  content((-1em, 0.7), [$c$])

  content((2.5, 1.5), [$p_v$])
  content((-7.8, 0.6), [$p_h$])
}))

When written as an equation for the optical depth, this becomes:

$
  tau = 2 sqrt((r + h_0)^2 - r^2) - sqrt((r + h_0)^2 - r^2 + r^2 cos^2 theta) - r cos theta
$

Where $theta$ is $theta_v$ for a view ray, and $theta_l$ for a light ray.
Note that this equation does not take the density at the viewer $v$ or midpoint $c$'s
altitude into account.

While this works for the optical depth used for transmittance, simply using
the scattering integral @eq:scatter-flat when looking down no longer works,
even when using the adjusted optical depth.

It is possible to split the scattering integral into two parts: one from the
view ray's closest point to the planet center, which can be calculated as normal,
and one from the viewer up to the closest point. This however gives the same
result visually as not splitting up the integral. This means another method of
calculating the scattering with a viewer from outside the atmosphere is needed.

== Views from space <chap:space-view>
Besides functioning as an approximation to single scattering, the integral for
the flat homogeneous atmosphere introduced in @eq:flat-atmo-integral can be
interpreted differently. Simplifying the integral, it becomes:

$ L_"segment" = tau_"segment" integral_0^1 exp(-t tau_a - (1-t) tau_b) $

Here, $tau_"segment"$ can be interpreted as the optical depth over a segment in
the atmosphere, $tau_a$ the optical depth to the light at the start of the
segment, and $tau_b$ the optical depth at the end of the segment.

The radiance that then reaches the viewer is $L_"segment"$ multiplied by the
transmittance from the viewer to the start of the segment. However, this can be
directly incorporated into the integral, by adding the optical depth from the
viewer to the start of the segment to both $tau_a$ and $tau_b$. The radiance
at the viewer from this segment is then:

$
  L_"viewer" = tau_"segment" F(theta_s) (exp(-tau_a - tau_"view") - exp(-tau_b - tau_"view")) / (tau_b - tau_a)
$

When this segment is used to represent the ray from the viewer towards the surface
of the planet, it can directly approximate the scattering on that segment.
This however does not work well from all viewer positions, especially when
the viewer is far above the planet surface, or is near sunrise.

To resolve this, it is instead possible to use the segment scattering to calculate
the scattering for a part of the view ray, and add them together. This new integration
method resembles the one introduced in @sebh-integrator, which is as follows:

$
  integral_0^d exp(-beta_a t) S dif t
$

Where $d$ is the ray length, and $S$ is the scattered light at this sample position.
This can be simplified to:

$
  (S - S exp(-beta_a d)) / beta_a
$

== Ozone layer <chap:ozone>
For ozone, @bruneton-new and @seb-skyatmo use the following density:

$ rho(h) = max(0, 1 - (|h - 25|) / 15) $

Where $h$ is in kilometers.
While an analytical integral is likely possible, a simple approximation for the
ozone layer can be set up as a constant density shell, starting at 17.5 kilometers
from the planet surface, and ending at 32.5 kilometers from the surface.

This is then expressed as follows:

$
  rho(h) = cases(
    1 "if" 17.5 <= h <= 32.5,
    0 "otherwise"
  )
$

The optical depth can then be calculated as the path length inside this shell,
which is the difference between the positions along the ray of the spheres
representing the bottom and top boundary of the shell.

== Multiple scattering
Due to time constraints, multiple scattering is not considered. It may be
possible to approximate it in a similar way as @underwater describes, as
this provides an analytical solution assuming a constant density medium.
This however does rely on an extra coefficient, $K_d$, which is a measured
value, and would have to be calculated as well.

Another method of implementing multiple scattering is done in @space-glider.
This depends on the delta-Eddington approximation, but not many details are
given in the original code.

= Evaluation <chap:eval>
In order to see if the new models hold up to the existing ones presented in
@chap:previous-work[chapter], they have to be compared to these models. This is done by
looking at the visual accuracy of the models to a path traced ground truth, as
well as looking at their runtime performance and implementation complexity.

== setup
As the intention is to run the atmosphere models on the GPU, the evaluation has
to take this into account.

For this purpose, a simple tool to run shaders is developed.
This shader runner uses the rust library `wgpu` for its GPU functionality.
`wgpu` can use multiple graphics API's such as Vulkan, and DirectX.

It can render a provided shader, in `SPIR-V` format, or `WGSL`, to either an output
texture, a buffer texture that can later be read by other shaders, or a volume
texture that can be read by other shaders. It can then outputs the output
textures to a specified image format. To preserve accuracy, the buffer textures
are in 32-bit floating point `RGBA` format, (`Rgba32Float` in `wgpu`) and may be
a 2D or 3D texture.

The tool can output images to the `png` file format, as well as the `exr`
image format, and numpy's `npy` file format @numpy. The `npy` format is used
as one of the image comparison tools fails to load the resulting `exr` images.

In order to measure the runtime of shaders, the shader runner can optionally
output timings recorded using `wgpu`'s timestamp queries. These report the
time difference from before and after issuing the draw call that renders the
shader in the render pipeline.

When compiled, the tool includes the shaders to compare directly into the executable,
so when the executable is run without any command-line arguments, it runs all the
included shaders and copies the performance statistics to the clipboard. This allows
for easier performance measurements on multiple computers. The link to the repository
containing all relevant code can be found in @chap:append-repo[appendix].

== Implemented models
In order to provide a good set of models to compare to, several of the models
mentioned in @chap:previous-work[Chapter] are implemented to compare against.

The parameters for the atmosphere were provided by the demo code from @seb-skyatmo,
and, when possible, the models have been set up so that they can use the parameters
as provided there. As these parameters provide a planet radius, a planet reflecting
no light is rendered as part of the atmosphere.

To keep the comparison simple, no spectral rendering is done, and it is assumed
the parameters provided are valid for a linear sRGB color space. If needed, the
models could be modified to provide spectral output, as described in @sky-comparison,
as well as @bruneton-new.

Some of the models have been ported to work in _shadertoy_ as well. Links to the
shadertoy shaders can be found in @chap:append-shaders[appendix].

=== Empty shader
This shader only outputs a white image, and does no other calculations. It is
used to measure any overhead in running the shader, besides the calculations that
the shader has to perform.

=== Path traced reference
To provide a ground truth to compare against, a path tracer is implemented,
according to @PBRT3e and @production-volume-rendering. It implements exponentially
decaying density for Mie and Rayleigh scattering, and a tent distribution for ozone
absorption. For each color channel, 4096 samples are taken and averaged per pixel.
4 scattering events are considered per pixel, and no denoising is applied.

=== Bruneton and Neyret
The implementation of the model presented in @bruneton-atmo is the one provided
in @bruneton-new, translated to work in the shader runner. It calculates 4
scattering orders. While the provided implementation allows calculating spectral
radiance as well, this is not used, as the parameters provided are for linear
sRGB instead.

=== Hillaire
This is an implementation of the model presented in @seb-skyatmo, ported from the
code supplied with the paper. It does not implement any of the frustum grid to
compute scattering, and instead falls back to raymarching when the camera is
outside of the atmosphere.

=== Preetham, Shirley and Smits
This is an implementation of the model presented in @preetham.
As this is a model already fitted to another atmosphere, it does not use any of
the parameters provided, and thus matches the resulting atmospheres less well.
It is also only capable of ground views, and thus cannot be used for comparisons
to higher altitude and space views.

=== Naive
The naive atmosphere is an implementation of the atmospheric scattering equations
by direct numerical integration. It is a modified version of @glsl-atmosphere,
adding ozone absorption in order to match the other implementations.

=== Schuler
This model is a derivation of the naive model, replacing the inner loop that
integrates the optical depth from the sample point towards the light, with
the Chapman function approximation as described in @gpu-pro-3. As the approximation
only works for an exponentially decaying atmosphere, this version of the model
does not incorporate ozone absorption. Note that while an implementation was provided
as supplementary material to @gpu-pro-3, this version could not be adapted to work
with the given parameters.

=== Neural network
This is an implementation of atmospheric scattering as described earlier in
@chap:neural-networks[chapter]. The parameters used are a result of one of several
training runs, as some training runs resulted in a less accurate fit.

=== Flat
This is an implementation of the spherical, homogeneous atmosphere as described
earlier in @chap:flat-atmo[chapter], as well as @cheap-sky. When looking down,
and the view ray hits the planet, the scattering is assumed to take place in a
segment, as described in @chap:space-view[chapter].

Note that the ozone layer is modeled as a constant density shell in this model,
instead of using the tent function the other models use. This results in a slight
mismatch in some view angles.


=== Raymarched
This is an implementation using the scattering integral as described in
@chap:space-view[chapter], as a replacement for the naive integrator used in
@glsl-atmosphere. For the optical depth, approximation from @chap:flat-atmo[chapter]
is used. The number of integration steps is set to 5, as this provides a good tradeoff
between quality and steps required.

At with the flat model, tho ozone layer for this model is also modeled as a constant
density shell, and may thus also appear different.

== Transmittance
Besides comparing the scattering results of the models, they can also be used to
calculate transmittance. As most models implemented use similar methods to calculate
transmittance, only the new neural network, and flat atmosphere models are considered.

=== Reference
This implements a naive Riemann integrator to calculate the optical depth. The
number of steps is intentionally set high, to 128 steps, to improve accuracy.

=== Neural network
This uses the transmittance neural network as described in @chap:neural-networks[chapter],
instead of the scattering neural network. When the view ray hits the planet,
the transmittance is calculated for the segment in a similar manner as @bruneton-atmo,
as the neural network only provides the transmittance for a full view ray.

=== Flat
This calculates the optical depth as described in @chap:flat-atmo[chapter], with
@eq:optical-depth-sphere[equation]:

#recall(<eq:optical-depth-sphere>)

The optical depth for the ozone layer is calculated as described in @chap:ozone[chapter].

If the view ray hits the planet surface, the optical depth for the segment between
the viewer and planet surface is computed in a similar manner as the optical depth
when looking down. However, the point on the planet surface is used as midpoint,
instead of the point closest to the planet center.

The optical depth is then used with Beer's law to calculate the transmittance:

$ T = exp(-tau) $

== viewer, light, and exposure
When comparing, all models are rendered from different viewer perspectives.
The following viewer positions are used:
- ground, directly on the planet surface.
- plane, at 5 kilometers above the planet surface.
- orbit, at 100 kilometers above the planet surface, then moved 1000 kilometers
  backwards, to give a better overview.
- space, showing the planet and atmosphere in full. The sun is positioned to the top
  right of the viewer, showing both the illuminated and dark side of the planet and
  atmosphere.

For the ground, plane and orbit views, the following times of day are used for
the viewer:
- Dawn, with the sun towards the viewer, at $6 degree$ below the horizon
- Sunrise, with the sun towards the viewer, at $6 degree$ above the horizon
- Noon, with the sun directly above the viewer.

A notable feature of the atmosphere is the planet shadow, where the planet itself
casts a shadow into the atmosphere. To view this, two extra viewing positions are
rendered as well, looking with the sun to the right side of the viewer, $3 degree$
below the horizon. One viewer is positioned on the planet surface, and one is
positioned 200 kilometers above the surface.

The sun, as light source, uses an irradiance of 6, for the `r`, `g` and `b`
color channels. No physical unit is used here.

For all view positions, a different camera exposure is used to map the radiance
to a final per-pixel color value that allows viewing the image without adjusting
the brightness later on. Exposure is implemented as simply multiplying the radiance
by the exposure value. For most view and light combinations, this value is $1$.
When the time of day is set to "dawn", an exposure of $16$ is used. For the "space"
viewpoint, an exposure of $2$ is used. For the planet shadow views, an exposure
of $32$ is used.

== Visual comparison
For all viewers and light combinations, each model is then rendered to an 1080
by 1920 images, and stored to disk as a `png` file, with $t(x) = 1.0 - exp(-x)$
applied per color channel as tonemap operator, and converted from linear sRGB
to nonlinear sRGB. Besides this, the image is also stored as an `exr` image, as
well as numpy's `npy` file format, storing the color channels directly as 32 bit
floating point numbers, in sRGB linear, without any other processing applied.

For comparing the results of the scattering models, the path tracing model is used
as reference. All other models are then compared to the path tracer, for each
viewer and light combination.

Comparison is done using nvidia-flip, in high dynamic range @flip-hdr.
As nvidia-flip is unable to load the `exr` images produced by the shader
tool, they are loaded in the `npy` file format. nvidia-flip in high dynamic range
also silently fails, and returns a mean error, as well as error map consisting of
only the value $0$ if there is any black pixel in the input image. For this reason,
a value of $1 dot 10^(-9)$ is added to the color channels of every pixel.
The mean error that nvidia-flip reports is then written out to a CSV file, and the
error map is written out to another image.

The second error metric that is used is the Root-Mean Square Error, or RMSE for short:

$ "RMSE" = sqrt(sum^N_(i = 0) (v_(i,m) - v_(i,t))^2 / N) $

Where $v_(i,m)$ is the $i$th sample of the model to test, and $v_(i,t)$ is the
$i$th sample of the reference model.

This is applied separately for each color channel, for each pixel in the image.
This is then converted into an image, and stored. The mean error for each image
is also stored.

This process is repeated for the transmittance models, where the reference
transmittance model is compared against the other transmittance models. However,
as the transmittance values range between $0$ and $1$, nvidia-flip is used with
low dynamic range instead @flip-ldr. Note that using nvidia-flip here is not
entirely correct, as the transmittance values are never displayed directly.
nvidia-flip however does provide a good error map to see where the transmittance
differs.

== Performance comparison
To measure runtime performance, `wgpu`'s timestamp queries are used. These record
the time at the point they are inserted into the render pipeline. In the shader
runner, one is inserted before the draw call to render the shader, and one after
this draw call. The difference between the two timestamps then corresponds to the
time it took the shader to run. To attempt to make the results more consistent,
the shader is run 512 times. The last 256 times, the timestamps are recorded
for profiling. The average time of these 255 runs is then used as time the model
runs in.

For the models presented in @bruneton-atmo and @seb-skyatmo, the time needed for
precomputation is recorded separately. However, for @seb-skyatmo, the scattering
table is assumed to be part of the final render, as this table needs to be
recomputed every time the viewer height, or the angle with the sun changes.

Hillaire's model is not able to use the lookup table when the viewer is outside
the atmosphere. This causes it to have different performance characteristics in
this case.

== Implementation complexity
The last method to compare models with is by implementation complexity. This is
a subjective measure of how complex the method is to implement both as a shader,
as well as any support code needed for precomputation. As a general rule applied
here, more lines of code indicate a more complex implementation. In this case,
the significant lines of code are counted. These are any line of code that satisfies
the following:
- is not empty.
- is not a comment.
- is not punctuation, such as `}` and `);`, which are common at the end of function
  calls and function definitions.
- contributes to the final output, and is not setup code, such as uniforms.

Requiring some kind of precomputation also increases complexity, as it requires
setting up extra code to render to textures, and then use these textures from
the shader. The difficulty of this depends on the renderer that the final model
is implemented in.

While the neural network based model does not do any precomputation in the same
way as the models presented in @bruneton-atmo and @seb-skyatmo, it does require
training a neural network, which adds to the complexity needed to make the final
model.

= Results
Here, the models are compared as described in the previous chapter.

== Visual
Here, the models are compared visually. For a full table of how each model
compares to the path traced reference, according to the nvidia-flip and RMSE
metrics, see @chap:append-flip and @chap:append-rmse. Due to space constraints,
not all models are compared to the path tracer for each view. Instead, the models
are only compared where the results differ more from the path tracer, as well as
when there is a failure mode of the model that can be highlighted.

=== Bruneton and Neyret, Hillaire
// makes comparing easy
#let triple = (place, types, names, placename) => {
  // automatically get the flip error metrics
  let flip-csv = csv("error-metrics/flip.csv")
  let header = flip-csv.first()
  let body = flip-csv.slice(1)

  // figure out the row
  let row = body.find(x => x.first() == place)

  // figure out the columns
  let col-l = header.position(x => x == types.at(0))
  let col-r = header.position(x => x == types.at(1))

  // errors, rounded
  let err-l = calc.round(float(row.at(col-l)), digits: 6)
  let err-r = calc.round(float(row.at(col-r)), digits: 6)

  figure(
    {
      show figure: set figure.caption(position: top)
      grid(
        columns: (auto, auto, auto),
        rows: (auto, auto, auto),
        gutter: 3pt,
        align: horizon,
        // names
        names.at(0), "Reference", names.at(1),
        // images
        image("comp/" + types.at(0) + "-" + place + ".png"),
        image("comp/" + "pathtrace" + "-" + place + ".png"),
        image("comp/" + types.at(1) + "-" + place + ".png"),
        // difference
        image("comp/diff/flip-" + types.at(0) + "-" + place + ".png"),
        stack("nvidia-flip errors:", spacing: 12pt, stack(
          dir: ltr,
          align(left)[$<-$ #err-l],
          align(right)[#err-r $->$],
        )),
        //[nvidia-flip errors: \ $<-$ #errs.at(0) \ #errs.at(1) $->$],
        image("comp/diff/flip-" + types.at(1) + "-" + place + ".png"),
      )
    },
    caption: placename,
    supplement: "Comparison",
  )
}

As both Bruneton and Neyret's and Hillaire's model implement multiple scattering,
they come very close to the reference path tracer.

Hillaire's model gives a slightly different halo around the sun, caused by Mie
scattering. This is likely due to the multiple scattering approximation assuming
the phase function is isotropic. This is most noticeable from the ground sunrise
view:

#triple(
  "ground-sunrise",
  ("bruneton", "skyatmo"),
  ("Bruneton and Neyret", "Hillaire"),
  "Ground",
)

This is less evident from a higher altitude:

#triple(
  "plane-sunrise",
  ("bruneton", "skyatmo"),
  ("Bruneton and Neyret", "Hillaire"),
  "Elevated viewer at 5km height from the surface",
)

However, Bruneton and Neyret's model does not handle the planet shadow viewed from
orbit very well, as there is a blocky pattern near the terminator. Hillaire's model
does not have this issue:

#triple(
  "planet-shadow-orbit",
  ("bruneton", "skyatmo"),
  ("Bruneton and Neyret", "Hillaire"),
  "From orbit",
)

=== Naive and Schuler
The Naive model and Schuler's model do not implement multiple scattering, and
will thus look different from the path traced reference. On top of this, Schuler's
model does not implement ozone absorption, and thus has different colors as well.

The lack of multiple scattering is evident as an overall darkened atmosphere:
#triple("ground-noon", ("naive", "schuler"), ("Naive", "Schuler"), "Ground")

The lack of ozone in Schuler's model becomes more apparent during sunrise, resulting
in the sky appearing less red:

#triple("ground-sunrise", ("naive", "schuler"), ("Naive", "Schuler"), "Ground")

The lack of ozone is visible even more when looking from space:

#triple("orbit-sunrise", ("naive", "schuler"), ("Naive", "Schuler"), "Ground")

The lack of multiple scattering becomes most evident when viewing the planet shadow
from the ground, as it illuminates the shadowed part of the atmosphere more, as seen
on the left in the images below:

#triple(
  "planet-shadow-ground",
  ("naive", "schuler"),
  ("Naive", "Schuler"),
  "Planet shadow, viewed from the ground",
)

=== Raymarched and Flat
The raymarched and flat models only implement single scattering. As the flat
model assumes it starts inside the atmosphere, it does not work well when the
viewer is outside the atmosphere.

Inside the atmosphere, they are very similar however:

#triple(
  "ground-sunrise",
  ("raymarched", "flat"),
  ("Raymarched", "Flat"),
  "Sunrise from the ground",
)

#triple(
  "ground-noon",
  ("raymarched", "flat"),
  ("Raymarched", "Flat"),
  "Noon from the ground",
)

The flat model still remains mostly similar at an elevated viewing position:
#triple(
  "plane-sunrise",
  ("raymarched", "flat"),
  ("Raymarched", "Flat"),
  "Sunrise from 5km above the surface",
)

#triple(
  "plane-noon",
  ("raymarched", "flat"),
  ("Raymarched", "Flat"),
  "Noon from 5km above the surface",
)

The flat model however breaks down from space views. Notice that the raymarched model
also has difficulty representing the transition into the shaded area of the planet
from this view, with the step count of $5$ used:

#triple(
  "space",
  ("raymarched", "flat"),
  ("Raymarched", "Flat"),
  "Far view from space",
)

When used from a viewer with lower elevation, the flat model still fails when viewed
from outside the atmosphere, and loses part of the color caused by ozone absorption:

#triple(
  "orbit-sunrise",
  ("raymarched", "flat"),
  ("Raymarched", "Flat"),
  "Sunrise from space",
)


With the low step count used, the raymarched model also does not represent the
planet shadow well. Note that the flat model fails completely here, as it does not
take the planet shadow into account at all:

#triple(
  "planet-shadow-orbit",
  ("raymarched", "flat"),
  ("Raymarched", "Flat"),
  "Planet shadow, viewed from 200km above the surface",
)

=== Raymarched and Naive
However, due to the improved integrator, the raymarched model does improve over
the naive model in some cases, such as when viewing the sunrise. Note how the
horizon is not darkened in the raymarched model, and more closely matched the
reference:

#triple(
  "plane-sunrise",
  ("raymarched", "naive"),
  ("Raymarched", "Naive"),
  "Sunrise viewed 5km from the surface",
)

The naive model has a noticeable darkening around the horizon, that both the
reference path tracer, and the raymarched model do not have. While this can be
resolved by increasing the number of integration steps in the naive model, the
improved integrator of the raymarched model does not require this.

=== Flat and Preetham, Shirley and Smits
As Preetham, Shirley and Smits' model is not fitted on the reference path tracer,
it will look different. However, it does provide a good baseline for a fitted model
that works only close to the ground. It still mostly gives the same image for a ground-bound
viewer:

#triple(
  "ground-sunrise",
  ("flat", "preetham"),
  ("flat", "Preetham, Shirley and Smits"),
  "Sunrise from the ground",
)

#triple(
  "ground-noon",
  ("flat", "preetham"),
  ("flat", "Preetham, Shirley and Smits"),
  "Noon from the ground",
)

However, as it's not made to work after sunset, it fails here. The flat model also
fails in this case, as it does not properly take the planet shadow into account:

#triple(
  "ground-dawn",
  ("flat", "preetham"),
  ("flat", "Preetham, Shirley and Smits"),
  "Sun below the horizon, from the ground",
)

Both also fail to represent the planet shadow:

#triple(
  "planet-shadow-ground",
  ("flat", "preetham"),
  ("Flat", "Preetham, Shirley and Smits"),
  "Planet shadow viewed from the ground",
)

=== Neural network
The new fitted model based on neural networks largely reproduces the right
colors of the reference path tracer, but does not correctly reconstruct details,
such as the Mie scattering halo:

#triple(
  "ground-sunrise",
  ("nn", "skyatmo"),
  ("Neural network", "Hillaire"),
  "Sunrise from the ground",
)

Or the planet shadow:

#triple(
  "planet-shadow-ground",
  ("nn", "skyatmo"),
  ("Neural network", "Hillaire"),
  "Planet shadow from the ground",
)

This is less noticeable during daytime:

#triple(
  "ground-noon",
  ("nn", "skyatmo"),
  ("Neural network", "Hillaire"),
  "Noon from the ground",
)

Due to the model not being able to reconstruct finer details, the neural network
model does not represent dawn very well:

#triple(
  "plane-dawn",
  ("nn", "skyatmo"),
  ("Neural network", "Hillaire"),
  "Sun below the horizon, 5km above the surface",
)

From an elevated viewing position, it also does not represent the light scattered
between the planet surface and the viewer well, and makes it too dark:

#triple(
  "plane-noon",
  ("nn", "skyatmo"),
  ("Neural network", "Hillaire"),
  "Noon, 5km above the surface",
)

When viewed from space, the neural network appears to make the edges of the
atmosphere thicker:

#triple(
  "space",
  ("nn", "skyatmo"),
  ("Neural network", "Hillaire"),
  "From space",
)

== Transmittance
For both the neural network model, as well as the new optical depth approximation,
the transmittance can be compared to a naive, raymarched reference, using 128 steps
of Riemann integration. The nvidia-flip and RMSE error metrics for each model are
found in @chap:append-tr-flip and @chap:append-tr-rmse. Note that as the transmittance
is not directly used as an output image, using nvidia-flip to judge similarity
may not be appropriate. In the comparisons below, nvidia-flip is only used to
create a visual difference map.

// makes comparing easy
#let transmit = (place, placename) => {
  // automatically get the flip error metrics
  let mse-csv = csv("error-metrics/mse-transmittance.csv")
  let header = mse-csv.at(0)
  let body = mse-csv.slice(1)

  // figure out the row
  let row = body.find(x => x.at(0) == place)

  // figure out the columns
  let col-l = header.position(x => x == "transmittance-nn")
  let col-r = header.position(x => x == "transmittance-flat")

  // errors, rounded
  let err-l = calc.round(float(row.at(col-l)), digits: 6)
  let err-r = calc.round(float(row.at(col-r)), digits: 6)

  figure(
    {
      show figure: set figure.caption(position: top)
      grid(
        columns: (auto, auto, auto),
        rows: (auto, auto, auto),
        gutter: 3pt,
        align: horizon,
        // names
        "Neural network", "Reference", "Optical depth approximation",
        // images
        image("comp/" + "transmittance-nn" + "-" + place + ".png"),
        image("comp/" + "transmittance-naive" + "-" + place + ".png"),
        image("comp/" + "transmittance-flat" + "-" + place + ".png"),
        // difference
        image("comp/diff/flip-transmittance-nn" + "-" + place + ".png"),
        stack("RMSE errors:", spacing: 12pt, stack(
          dir: ltr,
          align(left)[$<-$ #err-l],
          align(right)[#err-r $->$],
        )),
        image("comp/diff/flip-transmittance-flat" + "-" + place + ".png"),
      )
    },
    caption: placename,
    supplement: "Comparison",
  )
}

When visually inspecting the transmittance from the new approximation, it results
in slightly more absorption near the horizon. This is expected, as the approximation
gives a larger optical depth than the reference, as seen in @fig:optical-depth-plots[figure].

The neural network does noticeably worse, as it visually fails at giving the same colors.
This error is also reflected in a higher RMSE:
#transmit("ground-dawn", "Ground")
#transmit("plane-dawn", "5km from the ground")

From orbit, it becomes more clear the approximation uses a different way of
evaluating the optical depth for the ozone layer, as there is a harsh cutoff
that the reference does not have. From this view, it becomes more clear the neural
network mixes the ozone layer color into the rest of the atmosphere:
#transmit("orbit-dawn", "Ground")
#transmit("space", "From space")

== Performance
Here, the performance of the models is compared. As Hillaire's model has different
performance characteristics when inside and outside the atmosphere, due to not
using the lookup table outside the atmosphere, the performance measurements are
split up between inside the atmosphere and outside the atmosphere.

For performance comparisons, many GPU's from different vendors were tested. As
different GPU models from the same vendor do not exhibit completely different performance
characteristics, and to keep this comparison brief, only one GPU from each vendor
is represented here.

The performance data shown below shows the average runtime of each model, in
milliseconds, when rendering to a 1920 by 1080 `Rgba32Float` texture, for all views
inside, and outside the atmosphere. The viewpoint with the fastest time as well
as standard deviation (stdev) are provided as well.

The space view and planet shadow orbit view are not taken into account when plotting
the performance graphs. The reason for this is that they calculate the atmosphere for
a different amount of pixels than the ground, plane and orbit views, which results
in different runtimes.

#let perf-graph(gpu, path) = {
  let data = toml("perf/" + path)
  let names = (
    naive: "Naive",
    schuler: "Schuler",
    skyatmo: "Hillaire",
    raymarched: "Raymarched",
    bruneton: "Bruneton,\nNeyret",
    flat: "Flat",
    nn: "Neural network",
    preetham: "Preetham,\nShirley, Smits",
    empty: "Empty",
  )

  // average
  let add(l, r) = l + r
  let mean(x) = x.reduce(add) / x.len()

  // collect timings for the models
  let timings-in = names
    .keys()
    .map(key => {
      data
        .outputs
        .pairs()
        .filter(x => (
          x.first().starts-with(key)
            and (
              x.first().contains("-ground-") or x.first().contains("-plane-")
            )
        ))
        .map(x => (
          x.last().average
            * 1e-6 // skyatmo precompute depends on the camera position, so add it here
            + if x.first().contains("skyatmo") {
              data.passes.at(x.first()).average * 1e-6
            } else { 0 }
        ))
    })
    .map(mean)


  let timings-out = names
    .keys()
    .map(key => {
      data
        .outputs
        .pairs()
        .filter(x => (
          x.first().starts-with(key) and x.first().contains("-orbit-")
        ))
        .map(x => (
          x.last().average
            * 1e-6 // skyatmo precompute depends on the camera position, so add it here
            + if x.first().contains("skyatmo") {
              data.passes.at(x.first()).average * 1e-6
            } else { 0 }
        ))
    })
    .map(mean)

  // precompute table timings
  let precompute-bruneton = data
    .passes
    .pairs()
    .filter(x => x.first().contains("bruneton"))
    .map(x => x.last().total * 1e-6)
    .reduce((l, r) => l + r)

  let precompute-hillaire = data
    .passes
    .pairs()
    .filter(x => (
      x.first() == "skyatmo-transmittance"
        or x.first() == "skyatmo-multi-scatter"
    ))
    .map(x => x.last().total * 1e-6)
    .reduce((l, r) => l + r)

  figure(
    table(
      columns: 2,
      [*Model*], [*Time (ms)*],
      "Bruneton and Neyret", [#calc.round(precompute-bruneton, digits: 3)],
      "Hillaire", [#calc.round(precompute-hillaire, digits: 3)],
    ),
    caption: [#gpu, precompute times],
  )

  // draw the timings numbers
  let nums(timings, offset, label) = {
    timings
      .enumerate()
      .map(((x, y)) => {
        // put inside the bar if it's near the end
        let align = if (
          // for all timings
          y < 0.8 * (..timings-in, ..timings-out).reduce(calc.max)
        ) { left } else {
          right
        }

        lq.place(
          y,
          x + offset,
          pad(0.5em)[#calc.round(y, digits: 5)ms -- #label],
          align: align,
        )
      })
  }

  // hide the horizontal ticks
  show: lq.cond-set(lq.grid.with(kind: "y"), stroke: none)

  // atmosphere bar
  figure(
    lq.diagram(
      width: 100%,
      height: 344pt,
      yaxis: (
        ticks: names.values().enumerate(),
        subticks: none,
      ),
      xaxis: (label: "Runtime (ms)", subticks: none),
      lq.hbar(
        timings-in,
        range(timings-in.len()),
        offset: -0.2,
        width: 0.4,
        stroke: black,
        fill: silver,
      ),
      lq.hbar(
        timings-out,
        range(timings-in.len()),
        offset: 0.2,
        width: 0.4,
        stroke: black,
        fill: silver,
      ),

      // numbers
      ..nums(timings-in, -0.2, "in"),
      ..nums(timings-out, 0.2, "out"),
    ),
    caption: [#gpu timings],
  )
}

=== NVIDIA
Below are the tables for an NVIDIA RTX 3090, one with the viewer inside the
atmosphere, and one with the viewer outside the atmosphere. When the viewer is outside
the atmosphere, Hillaire's model switches to raymarching the atmosphere, and uses
a lookup table for its transmittance calculation. It also has a larger step count
(40) than Schuler's model (24), and thus becomes the second slowest model.

A somewhat unexpected result is that of Bruneton and Neyret's model being slower
than the raymarched model.

Below are the time it takes for precomputation, as well as a bar plot for the
average runtime for each view. "out" means the viewer is outside the atmosphere,
"in" means the viewer is inside the atmosphere:

#perf-graph("NVIDIA RTX 3090", "nvidia-3090-windows.toml")

=== AMD
On AMD GPU's, Bruneton and Neyret's model, as well as Hillaire's model, are
consistently faster than both the raymarched and neural network model, which is
not the case on NVIDIA GPU's. On an AMD RX 9070 XT, the flat model becomes faster
than both Hillaire's and Bruneton and Neyret's model, when the viewer is inside
the atmosphere. When the viewer is outside the atmosphere, Hillaire's model
becomes the second slowest again.

The timings for an AMD RX 7600 are below:

#perf-graph("AMD RX 7600", "amd-7600-linux.toml")

=== Steamdeck
As the steamdeck is a more common device with a lower end AMD GPU, it is included
as a separate diagram. It matches the performance characteristics of the other
AMD GPU's tested.

#perf-graph("Steamdeck GPU (AMD)", "steamdeck.toml")

=== Intel
Only one Intel integrated GPU was tested, which is the integrated GPU found in
the Intel Ultra 7 258V.

Of note is that when comparing the timings, the model by Preetham, Shirley and
Smits runs faster than the empty model, which is counterintuitive, as the empty
model should not perform more calculations than any of the actual models.

#perf-graph("Intel Ultra 7 258V iGPU", "intel-ultra-7-258v-linux.toml")

== Implementation complexity
When counting the significant lines of code for all implementation, this is the
result, ordered from least to most lines of code:
+ Preetham, Shirley and Smits (58)
+ Flat (58)
+ Raymarched (74)
+ Neural network (94)
+ Schuler (106)
+ Naive (118)
+ Path tracer (147)
+ Hillaire (208)
+ Bruneton and Neyret (832)

Assuming that reference code is available, this code would simply have to be
ported to the desired shader language and graphics framework, which is roughly
proportional to the lines of code. For both the models by Hillaire, as well as
Bruneton and Neyret, extra care has to be taken to properly set up the precomputation
steps, as well as required textures.

The precomputation of the model by Hillaire is quite simple, as only three lookup
tables are required, and they have a clear order in which they can be computed.

For Bruneton and Neyret's model, this is more difficult. Despite the excellent
documentation in @bruneton-new, the required precomputation steps are still not
thoroughly explained, and as it requires more lookup tables, as well as extra
intermediate steps to function. This makes setting up the precomputation more
error prone.

For the fitted models, both the model by Preetham, Shirley and Smits, as well
as the neural network model, the implementation complexity depends on whether
the given parameters in the reference code are desired or not. Both models require
some reference to fit the model to. This can be another atmosphere model,
@nishita93 in case of Preeham, Shirley and Smits' model, and the path tracer for
the neural network model.

For both fitted models, extra code is also needed to then fit the coefficients
to the reference data. In case of the neural network model, some manual work may
be required as well, as not all sets of coefficients fit the data as well as desired.

== Overall
Overall, Bruneton and Neyret's model, as well as Hillaire's model, are the closest
to the reference visually. While Preetham, Shirley and Smits' model is the fastest
of all models runtime wise, it is followed by both Bruneton and Neyret's model,
as well as Hillaire's model, when the viewer is inside the atmosphere. The flat
model, and the neural network model follow after this.

If multiple scattering is desired, both Bruneton and Neyret's model, as well as
Hillaire's model can be used, however, Hillaire's model may come with a performance
penalty when used for viewers outside the atmosphere. If no multiple scattering
is desired, the raymarched model, or Schuler's model can be used. If the viewer
remains in the atmosphere, the flat model provides an increase in performance,
at the cost of visually performing worse when the sun is below the horizon. The
naive model, using numerical integration, should not be used due to its high
performance cost.

According to the raymarched model, Schuler's model, and Hillaire's model when the
viewer is outside the atmosphere, using an approximation for the optical depth
provides a noticeable speedup over numerical integration. Using a lookup table
for transmittance results in a similar speedup.

While the neural network model provides good runtime performance, it does not do
well visually. This may be improved with a larger neural network, that would
also affect runtime performance.

= Conclusion
After comparing the implemented models, the questions from @chap:research-goals[chapter]
can now be answered.

The neural network model is a form of fitted model, and it works with both a viewer
near the ground, as well as a viewer in space. While performance differs between
GPU vendors, it remains close to lookup-table based models. The flat model remains
close in performance to the lookup-table based models as well, but the raymarched
model is slower, even with the low step count used.

The neural network model has to be retrained to change any of the atmosphere
parameters and properties. The flat and raymarched models, while not a fitted
model, can work directly with different parameters.

For both the flat and raymarched models, a different $rho(h)$ than an exponential
density is possible, as shown by the ozone layer used in both models. The approximation
used for the optical depth is also able to approximate a non-exponential density
profiles, according to @overview-chap. This however was not verified in the
comparisons done.

= Future work
The comparison done between Bruneton and Neyret's model, as well as Hillaire's model
is done using only the `Rgba32Float` texture format, with texture resolutions used
as provided in the reference implementations of these models. Of interest may be
using different texture formats, as well as resolutions to see if there is a
performance tradeoff in doing so. The transmittance lookup table from Hillaire's
model should also be compared against the optical dept approximation introduced,
as well as the one used in @gpu-pro-3.

The flat and raymarched models also do not model multiple scattering. This can
likely be added using a similar method to the one used in @underwater, or @space-glider.
The approximation used for the optical depth in both these models is also only
tested for an exponentially decaying density atmosphere. While @overview-chap
confirms it does work for different density profiles, they only test it with the
viewer at a fixed altitude. Extra work is likely required to make it work well
with a varying viewer altitude.

#bibliography("citations.bib", style: "apa", title: "References")

#show: appendix

#let flip-csv = csv("error-metrics/flip.csv")
#let rmse-csv = csv("error-metrics/mse.csv")

#let transmittance-flip-csv = csv("error-metrics/flip-transmittance.csv")
#let transmittance-rmse-csv = csv("error-metrics/mse-transmittance.csv")

#let err-table = (head, csv) => {
  let columns = csv.first().len()
  assert(
    columns == head.len(),
    message: "Header does not have the same column count as the csv file",
  )

  table(
    columns: columns,
    ..head,
    ..csv
      .slice(1)
      .map(x => {
        // row label
        let label = x.first()

        // position in the array of the lowest number
        let lowest = x
          .slice(1)
          .enumerate()
          .sorted(key: x => (x.at(1)))
          .at(0) // first
          .first()


        // numbers
        let nums = x
          .slice(1)
          .enumerate()
          .map(x => {
            let idx = x.at(0)
            let num = x.at(1)

            // to float
            let num = str(calc.round(float(num), digits: 8))

            // highlight?
            if idx == lowest { [*#num*] } else { num }
          })

        // return table
        (label, ..nums)
      })
      .flatten()
  )
}

#page(flipped: true, margin: 18pt)[
  = nvidia-flip errors <chap:append-flip>
  #align(center, err-table(
    (
      [*nvidia-flip errors*],
      [*Preetham, Shirley, \ and Smits*],
      [*Empty*],
      [*Raymarched*],
      [*Schuler*],
      [*Naive*],
      [*Hillaire*],
      [*Neural Net*],
      [*Flat*],
      [*Bruneton \ and Neyret*],
    ),
    flip-csv,
  ))

  = RMSE errors <chap:append-rmse>
  #align(center, err-table(
    (
      [*RMSE errors*],
      [*Preetham, Shirley, \ and Smits*],
      [*Empty*],
      [*Raymarched*],
      [*Schuler*],
      [*Naive*],
      [*Hillaire*],
      [*Neural Net*],
      [*Flat*],
      [*Bruneton \ and Neyret*],
    ),
    rmse-csv,
  ))
]

#pagebreak()


// select csv rows
#let csv-select-rename = (rows, csv) => {
  csv
    .filter(x => rows.keys().contains(x.first()))
    .map(x => (rows.at(x.first()), ..x.slice(1)))
}

= nvidia-flip errors, transmittance <chap:append-tr-flip>
#align(center, err-table(
  ([*nvidia-flip errors*], [*Neural network*], [*Flat*]),
  csv-select-rename(
    (
      view: "",
      ground-noon: "ground",
      plane-noon: "plane",
      orbit-noon: "orbit",
      space: "space",
      planet-shadow-orbit: "planet-shadow-orbit",
    ),
    transmittance-flip-csv,
  ),
))

= RMSE errors, transmittance <chap:append-tr-rmse>
#align(center, err-table(
  ([*RMSE errors*], [*Neural network*], [*Flat*]),
  csv-select-rename(
    (
      view: "",
      ground-noon: "ground",
      plane-noon: "plane",
      orbit-noon: "orbit",
      space: "space",
      planet-shadow-orbit: "planet-shadow-orbit",
    ),
    transmittance-rmse-csv,
  ),
))

= Link to code repository <chap:append-repo>
The public repository of the code used can be found here:
https://git.science.uu.nl/vig/mscprojects/a-visual-and-performance-comparison-of-atmospheric-scattering-models

= Links to shadertoy versions of shaders <chap:append-shaders>
Below are the links to the _shadertoy_ versions of the implemented shaders.
Only Bruneton and Neyret's atmosphere is not implemented in shadertoy, as it is
not possible to use 3D buffers.

- Flat: https://www.shadertoy.com/view/t3XBWH
- Raymarched: https://www.shadertoy.com/view/tXXBWH
- Neural Network: https://www.shadertoy.com/view/tXffD8
- Preetham, Shirley, Smits: https://www.shadertoy.com/view/WX2Bz1
- Naive: https://www.shadertoy.com/view/W3jBz1
- Schuler: https://www.shadertoy.com/view/WX2fR1
- Hillaire's: https://www.shadertoy.com/view/wcdGRX
- Path tracer: https://www.shadertoy.com/view/Wfd3D8

