#import "@preview/touying:0.6.1": *
#import "@preview/cetz:0.4.2"
#import "@preview/cetz-plot:0.1.3": plot
#import "@preview/lilaq:0.5.0" as lq
#import "theme.typ": *

#show "nvidia-flip": box(stack(dir: ltr, scale(x: -100%, "F"), "LIP"))

#let cetz-canvas = touying-reducer.with(
  reduce: cetz.canvas,
  cover: cetz.draw.hide.with(bounds: true),
)

#set text(font: "Fira Sans")
#show raw: set text(font: "Latin Modern Mono 12", size: 1.5em)

// UU colors
#let uu-red = rgb(192, 10, 53)
#let uu-yellow = rgb(255, 205, 0)
#let uu-dark-grey = rgb(64, 64, 64)
#let uu-almost-black = rgb(26, 26, 26)

#let uu-creme = rgb(249, 230, 173)
#let uu-orange = rgb(221, 149, 98)
#let uu-bordeaux = rgb(146, 30, 86)
#let uu-brown = rgb(99, 63, 43)
#let uu-green = rgb(99, 165, 147)
#let uu-blue = rgb(102, 135, 195)
#let uu-darkBlue = rgb(23, 29, 66)
#let uu-purple = rgb(82, 41, 128)


#show: metropolis-theme.with(
  aspect-ratio: "4-3",
  config-colors(
    primary: uu-red,
    primary-light: uu-dark-grey,
    secondary: uu-yellow,
    neutral-lightest: rgb("#fafafa"),
    neutral-light: uu-dark-grey,
    neutral-dark: uu-yellow,
    neutral-darkest: uu-almost-black,
  ),
  config-info(
    title: [A visual and performance comparison of atmospheric scattering models],
    author: [_Author:_ Dimas Leenman \ _1st Supervisor:_ Peter Vangorp \ _2nd Supervisor:_ Alex Telea],
    date: "29 Oct 2025",
    institution: [master Computing Science \ Utrecht University],
    logo: image("imgs/logo.png", height: 1em),
  ),
)

#let imgs(place) = {
  grid(
    columns: 2,
    gutter: 6pt,
    image("comp/" + place + "-ground-sunrise.png"),
    image("comp/" + place + "-planet-shadow-orbit.png"),

    image("comp/" + place + "-space.png"),
    image("comp/" + place + "-plane-noon.png"),
  )
}

#let perf-graph(gpu, path) = {
  let data = toml("perf/" + path)
  let names = (
    naive: "Naive",
    schuler: "Schuler",
    skyatmo: "Hillaire",
    raymarched: "Raymarched",
    bruneton: "Bruneton,\nNeyret",
    flat: "Flat",
    nn: "Neural network",
    preetham: "Preetham,\nShirley, Smits",
    empty: "Empty",
  )

  // average
  let add(l, r) = l + r
  let mean(x) = x.reduce(add) / x.len()

  // collect timings for the models
  let timings-in = names
    .keys()
    .map(key => {
      data
        .outputs
        .pairs()
        .filter(x => (
          x.first().starts-with(key)
            and (
              x.first().contains("-ground-") or x.first().contains("-plane-")
            )
        ))
        .map(x => (
          x.last().average
            * 1e-6 // skyatmo precompute depends on the camera position, so add it here
            + if x.first().contains("skyatmo") {
              data.passes.at(x.first()).average * 1e-6
            } else { 0 }
        ))
    })
    .map(mean)


  let timings-out = names
    .keys()
    .map(key => {
      data
        .outputs
        .pairs()
        .filter(x => (
          x.first().starts-with(key) and x.first().contains("-orbit-")
        ))
        .map(x => (
          x.last().average
            * 1e-6 // skyatmo precompute depends on the camera position, so add it here
            + if x.first().contains("skyatmo") {
              data.passes.at(x.first()).average * 1e-6
            } else { 0 }
        ))
    })
    .map(mean)

  // precompute table timings
  let precompute-bruneton = data
    .passes
    .pairs()
    .filter(x => x.first().contains("bruneton"))
    .map(x => x.last().total * 1e-6)
    .reduce((l, r) => l + r)

  let precompute-hillaire = data
    .passes
    .pairs()
    .filter(x => (
      x.first() == "skyatmo-transmittance"
        or x.first() == "skyatmo-multi-scatter"
    ))
    .map(x => x.last().total * 1e-6)
    .reduce((l, r) => l + r)

  // draw the timings numbers
  let nums(timings, offset, label) = {
    timings
      .enumerate()
      .map(((x, y)) => {
        // put inside the bar if it's near the end
        let align = if (
          // for all timings
          y < 0.8 * (..timings-in, ..timings-out).reduce(calc.max)
        ) { left } else {
          right
        }

        lq.place(
          y,
          x + offset,
          pad(0.5em)[#calc.round(y, digits: 5)ms -- #label],
          align: align,
        )
      })
  }

  // hide the horizontal ticks
  show: lq.cond-set(lq.grid.with(kind: "y"), stroke: none)
  show lq.selector(lq.diagram): set text(.8em)

  // atmosphere bar
  align(center, lq.diagram(
    width: 100%,
    height: 420pt,
    yaxis: (
      ticks: names.values().enumerate(),
      subticks: none,
    ),
    xaxis: (label: [Runtime (ms) -- #gpu], subticks: none),
    lq.hbar(
      timings-in,
      range(timings-in.len()),
      offset: -0.2,
      width: 0.4,
      stroke: black,
      fill: silver,
    ),
    lq.hbar(
      timings-out,
      range(timings-in.len()),
      offset: 0.2,
      width: 0.4,
      stroke: black,
      fill: silver,
    ),

    // numbers
    ..nums(timings-in, -0.2, "in"),
    ..nums(timings-out, 0.2, "out"),
  ))
}

#let perf-amd = perf-graph("AMD RX 7600", "amd-7600-linux.toml")
#let perf-nvidia = perf-graph("NVIDIA RTX 3090", "nvidia-3090-windows.toml")

#title-slide()

= Motivation
Why model the sky?

== Motivation
Games that take place outside may want to render a sky.

A simple gradient may be sufficient:

#image("imgs/minecraft.png")

== Motivation
A simple gradient may be sufficient.

But not for outside the atmosphere:

#image("imgs/scatterer.png")

== Motivation
Some games want both inside and outside the atmosphere:

#grid(
  columns: 2,
  gutter: 6pt,
  image("imgs/space-glider.png"), image("imgs/space-glider-2.png"),
)

= Physics
How does light interact with the sky?

== Scattering
#slide(repeat: 7, self => [
  #let (uncover, only) = utils.methods(self)

  #only(1, [This is a planet with an atmosphere:])
  #only(2, [And a viewer $v$ and light source $l$:])
  #only(3, [Light travels from $l$ through the atmosphere:])
  #only(4, [Then scatters towards $v$: Single scattering])
  #only(5, [Light can also scatter via other paths: Multiple scattering])
  #only(
    6,
    [Light can also scatter away, meaning not all light reaches the viewer],
  )
  #only(7, [How much light reaches the viewer depends on optical depth $tau$])

  #figure(cetz.canvas(length: 2.5em, {
    import cetz.draw: *
    let self = utils.merge-dicts(self, config-methods(
      cover: utils.method-wrapper(hide.with(bounds: true)),
    ))
    let (uncover,) = utils.methods(self)

    let ocontent(a, b, ..rest) = {
      content(
        a,
        {
          set text(size: 30pt)
          b
        },
        ..rest,
      )
    }

    stroke((thickness: 2pt))

    // planet
    arc-through((-5, -1), (0, 0), (5, -1))

    // atmosphere
    stroke((dash: "dashed"))
    arc-through((-7.5, 1), (0, 3), (7.5, 1)) // atmosphere
    stroke((dash: none))

    uncover("2-99", {
      // viewer
      fill(black)
      ocontent((-3.3, 0.7), [$v$])
      circle((-3, 1), radius: 0.1)
      fill(none)

      // sun
      stroke((dash: "dashed"))
      circle((5, 4), radius: 0.2)
      ocontent((5.4, 3.6), [$l$])
      stroke((dash: none))
    })

    uncover("3-99", {
      // scatter to sun
      line(
        (1, 1),
        (5, 4),
        mark: (start: ">", scale: 2, fill: black),
        name: "s-to-l",
      )
    })

    uncover("4-99", {
      // view to scatter
      line(
        (-3, 1),
        (1, 1),
        mark: (start: ">", scale: 2, fill: black),
        name: "v-to-s",
      )
    })

    uncover("5-6", {
      // scatter to sun, multiple
      stroke((dash: "dashed"))
      line((1, 1), (2, 1))
      line((2, 1), (3, 0))
      line((3, 0), (5, 4))
    })

    uncover("6-6", {
      // scatter to sun, multiple
      stroke((dash: "dashed"))
      line((0, 2), (-3, 4), mark: (
        end: ">",
        scale: 2,
        fill: black,
        stroke: (dash: none),
      ))
      line((0, 2), (5, 4))
    })

    uncover("7-99", {
      content(
        ("s-to-l.start", 40%, "s-to-l.end"),
        padding: 0.1,
        anchor: "south",
        angle: "s-to-l.end",
        [$tau$],
      )

      content(
        ("v-to-s.start", 50%, "v-to-s.end"),
        padding: 0.1,
        anchor: "south",
        angle: "v-to-s.end",
        [$tau$],
      )
    })
  }))
])

== Optical depth
Optical depth $tau$ depends on atmosphere density $rho(h)$,
where $h$ is the height from the planet center:

$ tau = integral_a^b rho(t^2 + h^2 + 2 h t cos theta_v) dif t $

== Single scattering event
Now, a radiance from a single scattering event can be calculated:

$ L_"scatter" = E_"light" F(theta_s) beta_s rho(h_s) T_s T_l $

Where $F(theta_s)$ is the phase function.

This is one of many scatter events in a view path

#figure(cetz.canvas(length: 2.5em, {
  import cetz.draw: *

  let ocontent(a, b, ..rest) = {
    content(
      a,
      {
        set text(size: 30pt)
        b
      },
      ..rest,
    )
  }

  stroke((thickness: 2pt))

  // planet
  arc-through((-5, -1), (0, 0), (5, -1))

  // atmosphere
  stroke((dash: "dashed"))
  arc-through((-7.5, 2), (0, 4), (7.5, 2)) // atmosphere
  stroke((dash: none))

  // viewer
  fill(black)
  ocontent((-3.3, 0.7), [$v$])
  circle((-3, 1), radius: 0.1)
  fill(none)

  // sun
  stroke((dash: "dashed"))
  circle((5, 4), radius: 0.2)
  ocontent((5.4, 3.6), [$l$])
  stroke((dash: none))

  // scatter to sun
  line(
    (1, 1),
    (5, 4),
    mark: (start: ">", scale: 2, fill: black),
    name: "s-to-l",
  )

  // view to scatter
  line(
    (-3, 1),
    (1, 1),
    mark: (start: ">", scale: 2, fill: black),
    name: "v-to-s",
  )

  content(
    ("s-to-l.start", 42%, "s-to-l.end"),
    padding: 0.1,
    anchor: "south",
    angle: "s-to-l.end",
    [$T_l = exp(-beta_a tau)$],
  )

  content(
    ("v-to-s.start", 50%, "v-to-s.end"),
    padding: 0.1,
    anchor: "south",
    angle: "v-to-s.end",
    [$T_s = exp(-beta_a tau)$],
  )
}))

= Reference model
The reference path tracer

== Path tracer
Path tracing closely approximates the path of photons inside the atmosphere:

#imgs("pathtrace")

== Path tracer
After 4096 samples per pixel, 4 scatter events:

#imgs("pathtrace")

== Path tracer
This takes 30 seconds per image on my desktop computer...

#imgs("pathtrace")

= Real time models
Now to do this in $<= 16$ milliseconds

#let double(place, type, name) = {
  // automatically get the flip error metrics
  let flip-csv = csv("error-metrics/flip.csv")
  let header = flip-csv.first()
  let body = flip-csv.slice(1)

  // figure out the row
  let row = body.find(x => x.first() == place)

  // figure out the columns
  let col = header.position(x => x == type)

  // errors, rounded
  let err = calc.round(float(row.at(col)), digits: 6)


  align(center, {
    show figure: set figure.caption(position: top)
    grid(
      columns: (auto, auto),
      rows: (auto, auto, auto),
      gutter: 6pt,
      align: horizon,
      // names
      "Reference", name,
      // images
      image("comp/" + "pathtrace" + "-" + place + ".png"),
      image("comp/" + type + "-" + place + ".png"),
      // difference
      align(right)[nvidia-flip error: #h(1em) #err $->$],
      image("comp/diff/flip-" + type + "-" + place + ".png"),
    )
  })
}

== Naive
What if we use numerical integration on the single scattering equation?

#imgs("naive")

== Naive
Works, but lacks multiple scattering:

#imgs("naive")

== Naive
This lack of multiple scattering is best noticeable near the planet shadow:
#double("planet-shadow-ground", "naive", "Naive")

== Naive
Daytime is also darker:

#double("plane-noon", "naive", "Naive")

== Naive
As is sunrise:

#double("plane-sunrise", "naive", "Naive")

== Naive
From the ground as well:

#double("ground-sunrise", "naive", "Naive")

== Naive
It's also slow:

#perf-amd

== Approximate the optical depth
Recall the equation for optical depth:
$ tau = integral_a^b rho(t^2 + h^2 + 2 h t cos theta_v) dif t $
#pause
Assuming $a = 0$, $b= infinity$ and $rho(h) = exp(-h / h_0)$, we can use Schuler's (2012) approximation:

#pause
$
  tau_"Ch" = cases(
    display(& exp(-h / h_0) c / (1 + c cos theta_v) & "if" theta_v < 90 degree),
    display(- & exp(-h / h_0) c / (1 - c cos theta_v) + 2 sqrt(x_0) exp(r / h_0 - x_0) quad & "otherwise")
  )
$
With
$
  c = sqrt((r + h) / h_0), quad "and" quad x_0 = sqrt(1 - cos^2 theta_v)((r + h) / h_0)
$

== Schuler's approximation
This works:

#imgs("schuler")

== Schuler's approximation
But also lacks multiple scattering:

#imgs("schuler")

== Schuler's approximation
The Ozone layer does not follow $rho(h) = exp(-h / h_0)$, so it's not included:

#double("orbit-sunrise", "schuler", "Schuler")

== Schuler's approximation
But it is faster:

#perf-amd

== Multiple scattering
Currently, multiple scattering is lacking.

#pause
Bruneton and Neyret (2008), precompute the scattering results into a texture.

#pause
This also allows them to store multiple scattering.

== Bruneton and Neyret
Now, multiple scattering works:

#imgs("bruneton")

== Bruneton and Neyret
Visually, this comes very close to the reference:

#double("planet-shadow-ground", "bruneton", "Bruneton and Neyret")

== Bruneton and Neyret
Visually, this comes very close to the reference:

#double("plane-sunrise", "bruneton", "Bruneton and Neyret")

== Bruneton and Neyret
Only noticeable artifacts are near the planet shadow from space:

#double("planet-shadow-orbit", "bruneton", "Bruneton and Neyret")

== Bruneton and Neyret
It's fast too:

#perf-amd

== Bruneton and Neyret
Is there a downside?

#pause
Yes, implementation complexity: 832 significant lines of code for the shader,
not counting the extra code needed for precomputation.

Compared to other models at roughly 150 significant lines of code.

== Hillaire
Hillaire (2020) tries to simplify this:

#pause
- Multiple scattering approximation instead of full computation.

#pause
- Scattering table is now dependent on viewer height and light direction.

#pause
- Viewer outside the atmosphere falls back to numerical integration.

== Hillaire
This looks as follows:

#imgs("skyatmo")

== Hillaire
Slightly less accurate than Bruneton and Neyret's model:

#double("ground-sunrise", "skyatmo", "Hillaire")

== Hillaire
Slower than Bruneton and Neyret's model when outside the atmosphere:

#perf-amd

== Fitted models
Instead of a lookup table, it is also possible to fit some formula to a reference model instead.

#pause
Of note:
Preetham et al (1999)
Single formula fitted to Nishita et al's (1993) model.

#pause
Hosek and Wilkie (2012)
Extended version of Preetham et al, to improve accuracy.

#pause
Wilkie et al (2021)
Canonical Polyadic Decomposition, closer to a compressed lookup table.
Their largest model is 2.2GB.

== Preetham et al
For Preetham, Shirley and Smits' model, this looks like this:

#imgs("preetham")

== Preetham et al
For this model, no viewer above the ground, or nighttime view is possible:

#imgs("preetham")

== Preetham et al
The implementation here is also not refitted on the reference:

#double("ground-sunrise", "preetham", "Preetham, Shirley and Smits")

== Preetham et al
It is the fastest model

#perf-amd

= Research goals
What can we find from this?

== Research goals
- Is it possible to make a fitted model that works from all perspectives?

#pause
- How well does this model perform compared to the other ones?

#pause
- How difficult is it to adapt this to use a non-exponential $rho(h)$

== Neural network
Can we find an equation that fits the lookup tables?

#pause
Symbolic regression -- Struggles on transmittance table

#pause
Kolmogorov-Arnold networks -- Struggles on transmittance table

#pause
Polynomial fit -- Need to put the polynomial inside another function, works for transmittance.

#pause
Neural networks -- 1 hidden layer of width 4 works well for transmittance.

#pause
Neural networks -- 2 hidden layers of width 8 works well for scattering.

== Neural network
This looks as follows:

#imgs("nn")

== Neural network
Sadly, not very accurate:

#double("ground-sunrise", "nn", "Neural network")

== Neural network
Sadly, not very accurate:

#double("planet-shadow-ground", "nn", "Neural network")

== Neural network
Ok during daytime:

#double("ground-noon", "nn", "Neural network")

== Neural network
Ok from space:

#double("space", "nn", "Neural network")

== Neural network
Quite fast too:

#perf-amd

== Flat model
Assume a flat homogeneous atmosphere:
#align(center, cetz.canvas(length: 2.5em, {
  import cetz.draw: *

  stroke((thickness: 2pt))

  // planet
  line((-8, 0), (8, 0))

  // atmosphere
  stroke((dash: "dashed"))
  line((-8, 3), (8, 3))

  // sun ray
  line((0, 1), (-5, 3))

  // sun
  circle((-6.5, 4), radius: 0.2)
  content((-6.1, 3.8), [$l$])
  stroke((dash: none))

  // viewer
  line((0, 3), (0, 1))
  content((1em, 2.5), [$h_v$])

  // view ray
  line((0, 1), (5, 3))

  // view angle
  arc((0, 2), start: 90deg, delta: -68deg, radius: 1)
  content((1em, 1.55), [$theta_v$])

  // sun angle
  arc((0, 2), start: 90deg, delta: 68deg, radius: 1)
  content((-1em, 1.55), [$theta_l$])

  // viewer point
  stroke((dash: none))
  fill(black)
  circle((0, 1), radius: 0.1)
  content((-1em, 0.9), [$v$])
}))
#pause
Which can be represented with:
$
  L_"scatter" = integral_0^h_v beta_s
  exp(-t beta_a / (cos theta_v)) exp(-(h_v - t) beta_a / (cos theta_l)) dif t
$
#pause
Which can be rewritten to:
$
  L_"scatter" = beta_s h_v (exp(-beta_a / (cos theta_v)) - exp(-beta_a / (cos theta_l))) / (beta_a / (cos theta_l) - beta_a / (cos theta_v)) dif t
$

== Flat model
Now, replace the flat atmosphere optical depth with something better:
$
  tau_"sphere" = beta_a sqrt((r + h_0)^2 - r^2 + r^2 cos^2 theta_v)
  - beta_a r cos theta_v
$

#pause
This is the optical depth of a constant density sphere of radius $r + h_0$:
#align(center, {
  // angle to go over
  let domain = lq.linspace(0, 1)

  // planet radius / scale height
  let r = 600

  // integral
  let p1 = x => {
    let tot = 0
    let n = 0
    let t = 0
    let dt = 1 / 2
    while n < 256 {
      tot += calc.exp(r - calc.sqrt(t * t + r * r + 2 * t * r * x)) * dt
      t += dt
      n += 1
    }
    tot
  }

  // DR-1
  let p2 = x => {
    calc.sqrt((r + 1) * (r + 1) - r * r + r * r * x * x) - r * x
  }

  // schuler
  let p3 = x => {
    let c = calc.sqrt(calc.pi * r / 2)
    c / (c * x + 1)
  }

  lq.diagram(
    width: 100%,
    height: 256pt,
    xaxis: (label: [$cos theta$]),
    yaxis: (label: [$tau$]),
    // integral
    lq.plot(
      domain,
      p1,
      label: "Integral",
      mark: none,
      smooth: true,
      color: black,
      stroke: (thickness: 2pt),
    ),

    // DR-1
    lq.plot(
      domain,
      p2,
      label: "Approximation",
      mark: none,
      smooth: true,
      color: black,
      stroke: (thickness: 2pt, dash: "dashed"),
    ),

    // schuler
    lq.plot(
      domain,
      p3,
      label: "Schuler",
      mark: none,
      smooth: true,
      color: black,
      stroke: (thickness: 2pt, dash: "dotted"),
    ),
  )
})

#pause
Shadertoy user Jodie has shown this to work.

== Flat model
This looks like this:

#imgs("flat")

== Flat model
It works well when the viewer is inside the atmosphere:

#double("ground-sunrise", "flat", "Flat")

== Flat model
Even works well when the viewer is elevated:

#double("plane-sunrise", "flat", "Flat")

== Flat model
But breaks from space:

#double("space", "flat", "Flat")

== Flat model
But breaks from space:

#double("orbit-sunrise", "flat", "Flat")

== Flat model
And at night:

#double("ground-dawn", "flat", "Flat")

== Flat model
But it is fast:

#perf-amd

== Raymarched
To fix the space view, split the view ray up in segments.

#pause
Then, apply the flat model to each segment.

== Raymarched
This fixes the space view:

#imgs("raymarched")

== Raymarched
With the low sample count, it does struggle near the planet shadow:

#double("planet-shadow-orbit", "raymarched", "Raymarched")

== Raymarched
And it's slower than the flat model:

#perf-amd

== General overview
- Naive -- Reasonably easy to implement, Slow, No multiple scattering.

#pause
- Schuler -- Reasonably easy to implement, Faster than Naive. No Ozone.

#pause
- Bruneton and Neyret -- Multiple scattering, Fast, very accurate, but complex.

#pause
- Hillaire -- Multiple scattering, Fast with viewer inside the atmosphere,
  very accurate, easier to implement.

#pause
- Preetham, Shirley and Smits -- Fastest, needs to be fitted, does not work for elevated viewer, or at dawn.

#pause
- Neural network -- Not very accurate, fast, needs to be fitted.

#pause
- Flat -- Fast, accurate when the viewer is inside the atmosphere

#pause
- Raymarched -- Slower, but works outside the atmosphere as well.


== Overall conclusion

#quote[
  - Is it possible to make a fitted model that works from all perspectives?
  - How well does this model perform compared to the other ones?
  - How difficult is it to adapt this to use a non-exponential $rho(h)$
]

#pause
- Yes, the neural network fitted model can work,
  and performance wise is similar to lookup table methods,
  but visually looks worse

#pause
- Neural network needs to be retrained when parameters change,
  different $rho(h)$ also needs that.

#pause
- Flat model is also fast,
  but needs more work to get a different $rho(h)$,
  and multiple scattering.

#pause
- Approximating or precomputing part of the numerical integration is worth it.

#focus-slide[Questions?]

== Nvidia
#perf-nvidia

