#version 460

// Path tracer, but for a fully dense sphere

layout(set = 0, binding = 15) uniform params {
    float solar_irradiance;
    float radius;
    float beta;
};

layout(set = 0, binding = 12) uniform view {
    vec3 cam_pos; // camera position
    vec3 sun_dir; // sun direction
    float depth; // number of bounces
};

layout(push_constant) uniform push {
    uint layer;
    uint width;
    uint height;
};

layout(location = 0) in noperspective centroid vec2 fragcoord;
layout(location = 0) out vec4 fragcolor;

const float PI = 3.14159265358979323846;

// pcg, see https://www.pcg-random.org/
uvec4 pcg4d(inout uvec4 v) {
    v = v * 1664525u + 1013904223u;
    v.x += v.y * v.w;
    v.y += v.z * v.x;
    v.z += v.x * v.y;
    v.w += v.y * v.z;
    v = v ^ (v >> 16u);
    v.x += v.y * v.w;
    v.y += v.z * v.x;
    v.z += v.x * v.y;
    v.w += v.y * v.z;
    return v;
}

// random number state
uvec4 random_state = uvec4(
    // pixel pos
    uvec2(fragcoord.xy * vec2(width, height)) + 1u,
    // what iteration we are on in the pass
    layer + 1u,
    // extra randomness
    uint(float(fragcoord.y + fragcoord.x) * (width + height)) + layer + 1u
);

// generate a random number
float random() {
    // https://experilous.com/1/blog/post/perfect-fast-random-floating-point-numbers
    return uintBitsToFloat((pcg4d(random_state).x >> 9u) | 0x3f800000u) - 1.0;
}

// uniform random direction in sphere
vec3 sample_uniform_dir() {
    float x = random();
    float y = random();
    float phi = 2.0 * PI * x;
    float cos_theta = 1.0 - 2.0 * y;
    float sin_theta = sqrt(1.0 - cos_theta * cos_theta);
    return vec3(sin_theta * cos(phi), sin_theta * sin(phi), cos_theta);
}

// adapted from: https://iquilezles.org/articles/spherefunctions/
vec2 bounds(vec3 ray, vec3 dir) {
    float b = dot(ray, dir);
    float c = dot(ray, ray) - radius * radius;
    float h = b * b - c;
    if (h < 0.0) return vec2(-1.0);
    return vec2(-b - sqrt(h), -b + sqrt(h));
}

// ratio tracking, to estimate transmittance
float ratio_tracking(vec3 start, vec3 dir) {
    // distance from start
    vec2 t_bounds = bounds(start, dir);

    // can be solved analytically
    return exp(-t_bounds.y * beta);
}

// scattering types
const uint I_MISS = 0u; // no scattering
const uint I_HIT = 1u; // scattering

// delta tracking, for the next scatter event
uint delta_tracking(vec3 start, vec3 dir, out float t) {
    t = 0.0;

    // maximum distance we can trace
    vec2 t_bounds = bounds(start, dir);

    // move to start
    if (t_bounds.y < t_bounds.x) return I_MISS;
    else t = max(t, t_bounds.x);

    // track
    float xi = random();
    t += -log(1.0 - xi) / beta;

    // outside the volume
    if (t > t_bounds.y) return I_MISS;

    // hit
    else return I_HIT;
}

// integrate
float integrate(vec3 start, vec3 dir) {
    // accumulated luminance
    float luminance = 0.0;
    
    // loop till end of depth
    for (uint i = 0u; i < uint(depth); i++) {
        // find the next hit
        float t;
        uint hit = delta_tracking(start, dir, t);

        // miss? stop
        if (hit == I_MISS) break;

        // hit? change direction
        else if (hit == I_HIT) {
            // add to luminance
            luminance
                += ratio_tracking(start + dir * t, normalize(sun_dir))
                * solar_irradiance
                * (4.0 / PI);
                
            // change direction
            vec3 new_dir = sample_uniform_dir();
            
            start += dir * t;
            dir = new_dir;
        }
    }

    return luminance;
}
            
void main() {
    // camera offset
    vec3 ray_start = cam_pos;

    // ray direction
    vec2 uv = (fragcoord - 0.5) * vec2(float(width) / float(height), 1.0);
    vec3 ray_dir = normalize(vec3(uv, 1.0));

    // grayscale only
    fragcolor.xyz = vec3(integrate(ray_start, ray_dir));
    fragcolor.w = 1.0;
}
