// shadertoy version of Preetham, Shirley and Smit's atmosphere, published at https://www.shadertoy.com/view/WX2Bz1

const vec3 cam_pos = vec3(0.0, 5.0, 0.0); // camera position
const vec3 sun_dir = normalize(vec3(0.0, 0.1, 1.0)); // sun direction
const vec3 solar_irradiance = vec3(4.0);

const vec3 UP = vec3(0.0, 1.0, 0.0); // up direction, +Y
const float PI = 3.14159265358979323846; // from bruneton/definitions.glsl
const float T = 2.0; // Hardcode 2 as turbidity

// model constants, from the appendix
// Y, x and y coefficients
const float A_Y =  0.1787 * T - 1.4630;
const float B_Y = -0.3554 * T + 0.4275;
const float C_Y = -0.0227 * T + 5.3251;
const float D_Y =  0.1206 * T - 2.5771;
const float E_Y = -0.0670 * T + 0.3703;

const float A_x = -0.0193 * T - 0.2592;
const float B_x = -0.0665 * T + 0.0008;
const float C_x = -0.0004 * T + 0.2125;
const float D_x = -0.0641 * T - 0.8989;
const float E_x = -0.0033 * T + 0.0452;

const float A_y = -0.0167 * T - 0.2608;
const float B_y = -0.0950 * T + 0.0092;
const float C_y = -0.0079 * T + 0.2102;
const float D_y = -0.0441 * T - 1.6537;
const float E_y = -0.0109 * T + 0.0529;

// chromacity matrices
// transpose due to GLSL having a different order
const mat4x3 x_z_consts = mat4x3(
    0.0017, -0.0290, 0.1169,
   -0.0037,  0.0638,-0.2120,
    0.0021, -0.0320, 0.0605,
    0.0000,  0.0039, 0.2589
);

const mat4x3 y_z_consts = mat4x3(
    0.0028, -0.0421,  0.1535,
   -0.0061,  0.0897, -0.2676,
    0.0032, -0.0415,  0.0667,
    0.0000,  0.0052,  0.2669
);

// perez formula
float F(float theta, float gamma, float A, float B, float C, float D, float E) {
    return (1.0 + A * exp(B / cos(theta))) * (1.0 + C * exp(D * gamma) + E * cos(gamma) * cos(gamma));
}

vec3 sky(vec3 ray_dir) {
    // view angle
    float cos_theta = dot(ray_dir, UP);
    float theta = acos(cos_theta);

    // sun-view angle
    float cos_gamma = dot(ray_dir, normalize(sun_dir));
    float gamma = acos(cos_gamma);

    // sun angle
    float cos_theta_s = dot(normalize(sun_dir), UP);
    float theta_s = acos(cos_theta_s);

    // zenith luminance
    float chi = ((4.0 / 9.0) - (T / 120.0)) * (PI - 2.0 * theta_s);
    float Y_z = (4.0453 * T - 4.9710) * tan(chi) - 0.2155 * T + 2.4192;

    // zenith chromacity
    float x_z = dot(vec3(T * T, T, 1.0), x_z_consts * vec4(theta_s * theta_s * theta_s, theta_s * theta_s, theta_s, 1.0));
    float y_z = dot(vec3(T * T, T, 1.0), y_z_consts * vec4(theta_s * theta_s * theta_s, theta_s * theta_s, theta_s, 1.0));
    
    // luminance
    float Y = Y_z * F(theta, gamma, A_Y, B_Y, C_Y, D_Y, E_Y) / F(0.0, theta_s, A_Y, B_Y, C_Y, D_Y, E_Y);
    
    // chromacity
    float x = x_z * F(theta, gamma, A_x, B_x, C_x, D_x, E_x) / F(0.0, theta_s, A_x, B_x, C_x, D_x, E_x);
    float y = y_z * F(theta, gamma, A_y, B_y, C_y, D_y, E_y) / F(0.0, theta_s, A_y, B_y, C_y, D_y, E_y);

    // to XYZ
    // https://en.wikipedia.org/wiki/CIE_1931_color_space#CIE_xy_chromaticity_diagram_and_the_CIE_xyY_color_space
    float X = (Y / y) * x;
    float Z = (Y / y) * (1.0 - x - y);

    // to linear sRGB (D65)
    // http://brucelindbloom.com/index.html?Eqn_RGB_XYZ_Matrix.html
    return transpose(mat3(
         3.2404542, -1.5371385, -0.4985314,
        -0.9692660,  1.8760108,  0.0415560,
         0.0556434, -0.2040259,  1.0572252
    )) * vec3(X, Y, Z);
}

void mainImage(out vec4 fragcolor, vec2 fragcoord) {
    // camera offset
    // no need to offset here due to it being a skydome model
    vec3 ray_start = cam_pos;

    // ray direction
    vec2 uv = (fragcoord / iResolution.xy - 0.5) * vec2(float(iResolution.x) / float(iResolution.y), 1.0);
    vec3 ray_dir = normalize(vec3(uv, 1.0));

    // only if the angle is > 0
    // multiply by 1 / 683, as that's the conversion factor between lumen to watts
    fragcolor = dot(ray_dir, UP) > 0.0
        ? vec4(sky(ray_dir) * (1.0 / 683.0) * solar_irradiance, 1.0)
        : vec4(0.0, 0.0, 0.0, 1.0);

    // gamma
    fragcolor.xyz = pow(fragcolor.xyz, vec3(1.0 / 2.2));
}

