// shadertoy version of the neural network atmosphere, published at https://www.shadertoy.com/view/tXffD8

const float bottom_radius = 6360.0;
const float top_radius = 6460.0;

const vec3 solar_irradiance = vec3(4.0);

const vec3 cam_pos = vec3(0.0, 5.0, 0.0); // camera position
const vec3 sun_dir = normalize(vec3(0.0, 0.1, 1.0)); // sun direction

vec3 sigmoid(vec3 x) {
    return 1.0 / (1.0 + exp(-x));
}

vec4 sigmoid(vec4 x) {
    return 1.0 / (1.0 + exp(-x));
}

vec3 transmittance_nn(float height, float cos_theta) {
    // input
    vec2 x0 = vec2(height, cos_theta);

    // layer 1
    vec4 x1 = sigmoid(
        mat2x4(0.0214, -0.0301, 0.0167, -0.0137, -7.7263, 0.3688, -23.9130, 27.3124)
        * x0 + vec4(0.2639, 0.9069, 2.9238, -3.3962)
    );

    // layer 2
    return sigmoid(
        mat4x3(5.1255, 6.4604, 5.3080, -10.4257, -12.4410, -19.4679, 5.5411, 6.7883, 8.2314, -13.7020, -16.9345, -24.2136)
        * x1 + vec3(-1.3815 ,-3.5609 ,0.0589)
    );
}

vec3 scattering_nn(float height, float cos_theta, float cos_light, float cos_gamma) {
    // input
    vec4 x0 = vec4(height, cos_theta, cos_light, cos_gamma);

    // layer 1
    vec4 x1_0 = sigmoid(
        mat4x4(-0.0172, -0.1585, 0.0333, -0.0285, 22.3921, -0.2200, 1.4461, 25.3310, -0.0007, -0.0781, 0.1343, 0.0405, -0.0648, 0.0780, -0.3812, 0.0019)
        * x0 + vec4(-0.6036, -0.7347, -0.1642, -1.1600)
    );
    vec4 x1_1 = sigmoid(
        mat4x4(0.0037, -0.0006, -0.0144, 0.0025, -2.6395, 0.1363, 35.6077, 3.8818, -0.4070, -4.3260, 0.1487, 4.8999, 0.4393, -0.0526, -0.0515, -5.9698)
        * x0 + vec4(-1.2288, -2.2124, -0.0969, -6.3626)
    );

    // layer 2
    vec4 x2_0 = sigmoid(
        mat4x4(-3.6339, -1.7529, -160.8163, -1.2848, 2.3444, 4.6788, 0.3880, 0.5233, 0.1393, 0.4772, 0.6404, 1.2324, 1.9955, 3.0016, -96.2228, -1.2282) * x1_0
        + mat4x4(0.8563, -10.3158, 10.8031, 0.9464, 32.1428, 6.0045, -19.5529, 5.2667, 2.1930, 1.2001, 6.2481, -1.3785, -4.7430, 1.5145, 3.7958, -0.7921) * x1_1
        + vec4(-5.7566, -3.9390, -3.9847, -0.7349)
    );

    vec4 x2_1 = sigmoid(
        mat4x4(-7.8952, -3.5147, -2.8325, -3.5076, -0.4839, -1.4481, 3.8031, 42.6138, 5.0020, 4.6249, 1.0171, 1.1628, -1.0066, -6.9750, 4.4482, -1.1279) * x1_0
        + mat4x4(5.6382, 2.8548, -10.0346, -10.7135, 1.1769, -5.9692, -7.9183, -0.9696, 1.3309, 3.4965, 3.7415, 9.3012, -0.8607, -1.1860, -2.3595, -1.5907) * x1_1
        + vec4(-2.1066, -4.0293, -3.1043, -18.8468)
    );

    // layer 3
    return exp(
        mat4x3(-10.5722, -14.9728, -15.4197, -5.1958, -5.2867, -5.5605, -0.9151, -0.9667, -1.0897, -2.0293, -3.0072, -7.4153) * x2_0
        + mat4x3(0.0178, 1.6596, 5.6463, -13.0418, -12.7041, -12.4718, -3.0943, -2.1226, -0.9017, -27.3160, -20.7745, -18.8013) * x2_1
        + vec3(-1.1564 ,-1.2119 ,-1.2709)
    );
}

// adapted from: https://iquilezles.org/articles/spherefunctions/
vec2 atmo_bounds(vec3 ray, vec3 dir) {
    float b = dot(ray, dir);
    float c = dot(ray, ray) - top_radius * top_radius;
    float h = b * b - c;
    if (h < 0.0) return vec2(-1.0);
    return vec2(-b - sqrt(h), -b + sqrt(h));
}

// where does the planet start?
// adapted from: https://iquilezles.org/articles/spherefunctions/
vec2 planet_bounds(vec3 ray, vec3 dir) {
    float b = dot(ray, dir);
    float c = dot(ray, ray) - bottom_radius * bottom_radius;
    float h = b * b - c;
    if (h < 0.0) return vec2(-1.0);
    return vec2(-b - sqrt(h), -b + sqrt(h));
}

void mainImage(out vec4 fragcolor, vec2 fragcoord) {
    // camera offset
    vec3 ray_start = cam_pos + vec3(0.0, bottom_radius, 0.0);

    // ray direction
    vec2 uv = ((fragcoord / iResolution.xy) - 0.5) * vec2(float(iResolution.x) / float(iResolution.y), 1.0);
    vec3 ray_dir = normalize(vec3(uv, 1.0));
    // bounds
    vec2 t_start = atmo_bounds(ray_start, ray_dir);
    vec2 t_end = planet_bounds(ray_start, ray_dir);

    // stop if no hit
    if (t_start.y <= 0.0) fragcolor.xyz = vec3(0.0);
    else if (t_end.y <= 0.0) {
        // single hit
        // move to boundary
        ray_start += ray_dir * max(t_start.x, 0.0);

        // transmittance
        float height = length(ray_start) - bottom_radius;
        float cos_theta = dot(ray_dir, normalize(ray_start));
        float cos_light = dot(sun_dir, normalize(ray_start));
        float cos_gamma = dot(ray_dir, sun_dir);
        
        fragcolor.xyz = scattering_nn(height, -cos_theta, cos_light, cos_gamma) * solar_irradiance;
    } else {
        // hit the planet
        // move to boundary
        vec3 ray_start_n = ray_start + ray_dir * max(t_start.x, 0.0);

        // transmittance
        float height = length(ray_start_n) - bottom_radius;
        float cos_theta = dot(ray_dir, normalize(ray_start_n));
        float cos_light = dot(sun_dir, normalize(ray_start_n));
        float cos_gamma = dot(ray_dir, sun_dir);

        vec3 transmittance_viewer = transmittance_nn(height, cos_theta);
        fragcolor.xyz = scattering_nn(height, -cos_theta, cos_light, cos_gamma) * solar_irradiance;

        // move to planet boundary
        vec3 ray_start_f = ray_start + ray_dir * max(t_end.x, 0.0);

        // transmittance
        height = length(ray_start_f) - bottom_radius;
        cos_theta = dot(ray_dir, normalize(ray_start_f));
        cos_light = dot(sun_dir, normalize(ray_start_f));
        cos_gamma = dot(ray_dir, sun_dir);

        vec3 transmittance_surface = transmittance_nn(height, cos_theta);
        fragcolor.xyz -= scattering_nn(height, -cos_theta, cos_light, cos_gamma)
            * solar_irradiance * (transmittance_surface / transmittance_viewer);
    }
    
    // gamma
    fragcolor.xyz = pow(fragcolor.xyz, vec3(1.0 / 2.2));

    // ensure range
    fragcolor.xyz = clamp(fragcolor.xyz, vec3(0.0), vec3(1.0));
}

